"""
"mapping": {
    "default": "hamster",
    "slots": {
        "prs_ops": {
            "enable": true,
            "container": {
                "memory_limit": "3477078016",
                "cpu_policy": "normal",
                "cpu_limit": "6.36630754163",
                "memory_guarantee": "524288000",
                "cpu_guarantee": "6.36630754163",
                "recharge_on_pgfault": false
            },
            "port": 15695
        },
        "hamster": {
            "enable": true,
            "container": {
                "memory_limit": "3477078016",
                "cpu_policy": "normal",
                "cpu_limit": "6.36630754163",
                "memory_guarantee": "524288000",
                "cpu_guarantee": "6.36630754163",
                "recharge_on_pgfault": false
            },
            "port": 15694
        }
    }
}
"""
import logging
import os

import plugin

import multibeta2.balancer
import multibeta2.basesearch
import multibeta2.mapping
import utils


class Plugin(plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard', help='Shard', required=True)
        parser.add_argument('--httpsearch', help='Basesearch', required=True)
        parser.add_argument('--httpsearch_config', help='Config for httpsearch', required=True)
        parser.add_argument('--models', help='Models.archive', required=True)
        parser.add_argument('--db-timestamp', help='DB timestamp', required=True)
        parser.add_argument('--cache-dir', help='Fast storage (for caches)', required=False)
        parser.add_argument('--extra-arguments', help='extra args to daemon', required=False)
        parser.add_argument('-V', help='extra args to daemon (forwarded as -V)', action='append', dest='v_arguments')

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)

        self._host = host
        self._baseport = int(port)
        self._shard = os.path.abspath(args.shard)
        extra_arguments_list = ["-V {}".format(x) for x in args.v_arguments] if args.v_arguments is not None else []
        if args.extra_arguments is not None:
            extra_arguments_list.append(args.extra_arguments)

        class InstanceFactory(multibeta2.mapping.get_search_class(utils.get_itype(), utils.get_metaprj())):
            host = self._host
            shard = self._shard
            db_timestamp = args.db_timestamp
            cache_root = args.cache_dir
            extra_arguments = " ".join(extra_arguments_list)

        self._httpsearch_class = InstanceFactory

        self._balancer = multibeta2.balancer.Balancer(self._host, self._baseport, self._baseport + 7)
        self._slots = {}
        self._config = None

        self._httpsearch = os.path.abspath(args.httpsearch)
        self._httpsearch_config = os.path.abspath(args.httpsearch_config)
        self._models = os.path.abspath(args.models)
        self._cache_dir = args.cache_dir

        self._generate_resources_definition()

    def collect_status(self):
        return {
            'slots': {
                slot_name: searcher.collect_status()
                for slot_name, searcher in self._slots.items()
            },
            'balancer': self._balancer.collect_status(),
        }

    def apply_config(self, content):
        for slot_name, slot_data in content['mapping']['slots'].items():
            if slot_data['enable'] and slot_name not in self._slots:
                if self._cache_dir:
                    slot_cache_dir = os.path.join(self._cache_dir, str(slot_data['port']))
                    utils.ensure_dir(slot_cache_dir)
                httpsearch = self._httpsearch_class(slot_data['port'])
                httpsearch.apply_config({'conf_hash': slot_name,
                                         'resources': [self._resources]})
                httpsearch.run(
                    self._resources,
                    balancer_port=self._baseport,
                    container_properties=slot_data.get('container', {}),
                )
                self._slots[slot_name] = httpsearch

        balancer_mapping = _generate_mapping(content)
        logging.debug('{}'.format(balancer_mapping))
        self._balancer.apply_config(balancer_mapping)

        self._config = content

    def stop(self):
        self._balancer.stop()
        for slot_name, httpsearch in self._slots.items():
            httpsearch.stop()

    def _generate_resources_definition(self):
        self._resources = dict.fromkeys(self._httpsearch_class.resources_required)

        for k in self._resources:
            if k.endswith('.executable'):
                self._resources[k] = self._httpsearch
            if k.endswith('.models'):
                self._resources[k] = self._models
            if k.endswith('.cfg'):
                self._resources[k] = self._httpsearch_config


def _generate_mapping(config):
    mapping = {
        slot_name: slot_data['port']
        for slot_name, slot_data in config['mapping']['slots'].items() if slot_data['enable']
    }

    default_searcher = config['mapping']['default']
    if default_searcher:
        if default_searcher in config['mapping']['slots']:
            mapping_default_port = config['mapping']['slots'][config['mapping']['default']].get('port')
            if mapping_default_port:
                mapping['default'] = mapping_default_port
        elif default_searcher == 'error503':  # special default, for example error503
            return {}

    return mapping
