import collections
import functools
import logging

import infra.callisto.libraries.container as container

import plugin


class Plugin(plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--program', help='Program to run', required=True)
        parser.add_argument('--shard', help='Shard', required=False)
        parser.add_argument('--extra', help='Extra arg (multiple)',
                            action='append', default=list(), type=lambda arg: arg.split('='))

    # args: {program, shard} and all other command line args
    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self.workloads = collections.defaultdict(_get_daemon_factory(self.itype, self.metaprj, args))

    def apply_config(self, config):
        config = _parse_config(config)

        # stop what must be stopped
        absent = set(self.workloads.keys()) - set(workload.meta.name for workload in config.workloads)
        for workload in absent:
            self._stop_workload(workload)

        # (re)start others
        for workload in config.workloads:
            self._start_workload(workload)

    # TODO: return successfully applied config id (rev, hash, whatever)
    # TODO: and / or config being applied
    def collect_status(self):
        return {'workloads': [workload.collect_status() for workload in self.workloads.itervalues()]}

    def _stop_workload(self, workload):
        if workload not in self.workloads:
            _logger.error('Attempt to stop nonexistent workload [%s]', workload)
        else:
            _logger.info('Stop workload [%s]', workload)
            self.workloads.pop(workload).stop()

    def _start_workload(self, workload):
        self.workloads[workload.meta.name].apply_config(workload)


class Daemon(object):
    def __init__(self, program, shard):
        self.program = program
        self.shard = shard
        self.config = None
        self.container = None

    @property
    def port(self):
        return self.config.spec.port

    @property
    def name(self):
        return '{}_{}'.format(self.config.meta.name, self.port)

    def apply_config(self, config):
        self.config = config
        self.container = container.ContainerRunner(self.name)
        self.run()

    def shutdown(self):
        """Graceful shutdown"""
        pass

    def collect_status(self):
        """Default: check label in container, check port"""
        pass

    def run(self):
        assert self.config

        self.shutdown()

        # properties = {'labels[AGENT.conf_hash]': self.config.meta.hash}
        properties = {}
        properties.update(self.config.spec.container)

        self.container.run(self.command, properties=properties)

    def stop(self):
        pass

    @property
    def command(self):
        raise NotImplementedError('`command` property must be implemented in order to run Daemon')


class InvertedIndex(Daemon):
    def __init__(self, program, shard, db_timestamp=None):
        super(InvertedIndex, self).__init__(program, shard)
        self.db_timestamp = db_timestamp

    @property
    def command(self):
        args = [self.program,
                '--port', self.port,
                '--index-dir', self.shard] + self.config.spec.arguments
        if self.db_timestamp is not None:
            args.extend(["--db-timestamp", self.db_timestamp])
        return args


class Embedding(Daemon):
    def __init__(self, program, shard, models, db_timestamp=None):
        super(Embedding, self).__init__(program, shard)
        self.models = models
        self.db_timestamp = db_timestamp

    @property
    def command(self):
        args = [self.program,
                '--port', self.port,
                '--index-dir', self.shard,
                '--models-archive', self.models] + self.config.spec.arguments
        if self.db_timestamp is not None:
            args.extend(["--db-timestamp", self.db_timestamp])
        return args


class Basesearch(Daemon):
    def __init__(self, program, shard, models, remote_storage_config=None):
        super(Basesearch, self).__init__(program, shard)
        self.models = models
        self.remote_storage_config = remote_storage_config

    @property
    def command(self):
        cmd = [self.program,
                '-p', self.port,
                '-V', 'IndexDir={}'.format(self.shard),
                '-V', 'MXNetFile={}'.format(self.models)] + self.config.spec.arguments
        if self.remote_storage_config:
            cmd += ['-V ', 'RemoteIndexArcConfigPath={}'.format(self.remote_storage_config)]
        return cmd


class RemoteStorage(Daemon):
    def __init__(
            self, program, shard, **kwargs
    ):
        super(RemoteStorage, self).__init__(program, shard)
        self.program = program
        self.kwoptions = kwargs

    @property
    def command(self):
        result = [self.program, '--port', self.port]
        for k, v in self.kwoptions.items():
            k = k.replace('_', '-')  # compatibility with old services
            result += ['--' + k, v]
        return result + self.config.spec.arguments

Config = collections.namedtuple('Config', ['meta', 'workloads'])
Workload = collections.namedtuple('Workload', ['meta', 'spec'])
WorkloadMeta = collections.namedtuple('WorkloadMeta', ['name', 'hash', 'tags'])
WorkloadSpec = collections.namedtuple('WorkloadSpec', ['arguments', 'container', 'port'])


def _parse_config(config):
    workloads = [_parse_workload(workload) for workload in config['workloads']]
    assert len(workloads) == len(set(workload.meta.name for workload in workloads))
    return Config(config['meta'], workloads)


def _parse_workload(workload):
    meta = workload['meta']
    spec = workload['spec']
    return Workload(WorkloadMeta(meta['name'], meta['hash'], meta['tags']),
                    WorkloadSpec(spec['arguments'], spec['container'], spec['port']))


# TODO: separate cmdline args from formal parameters
def _get_daemon_factory(itype, metaprj, settings):
    cls = _TAGS_TO_CLASS.get((itype, metaprj)) or _TAGS_TO_CLASS[(itype, )]
    return functools.partial(cls, settings.program, settings.shard, **dict(settings.extra))


_TAGS_TO_CLASS = {
    ('invindex', ): InvertedIndex,
    ('embedding', ): Embedding,
    ('base', ): Basesearch,
    ('mmeta', 'imgs'): Daemon,  # for example
    ('remotestorage', ): RemoteStorage,
}


CONFIG_EXAMPLE = """
{
  "meta": {
    "instance": "vla1-6260:32322",
    "topology": {
      "group": "VLA_WEB_TIER1_INVERTED_INDEX",
      "version": "trunk"
    }
  },
  "workloads": [
    {
      "meta": {
        "hash": "304517317b53e58760da558ec7f2815b",
        "name": "hamster",
        "tags": {
          "ctype": "hamster"
        }
      },
      "spec": {
        "arguments": [
          "--server-config",
          "vla1-6260:32322_hamster_server.cfg",
          "--inverted-index-config",
          "vla1-6260:32322_hamster_invindex.cfg"
        ],
        "container": {
          "cpu_guarantee": 1.5754233950374164,
          "cpu_limit": 3.1508467900748327,
          "cpu_policy": "normal"
        },
        "port": 32323
      }
    },
    {
      "meta": {
        "hash": "2e1925df3573834cc7098559a34ca457",
        "name": "prs_ops",
        "tags": {
          "ctype": "hamster"
        }
      },
      "spec": {
        "arguments": [
          "--server-config",
          "vla1-6260:32322_hamster_2_server.cfg",
          "--inverted-index-config",
          "vla1-6260:32322_hamster_2_invindex.cfg"
        ],
        "container": {
          "cpu_guarantee": 1.7723513194170935,
          "cpu_limit": 3.465931469082316,
          "cpu_policy": "rt"
        },
        "port": 32327
      }
    }
  ]
}
"""

_logger = logging.getLogger(__name__)
