import os
import logging

import gevent.pywsgi

import v2.plugin
import infra.callisto.libraries.container as container_lib


class Instance(object):
    def __init__(self, host, port, binary_path, config_path, shard_path):
        self.host = host
        self.port = port
        self.binary_path = binary_path
        self.config_path = config_path
        self.shard_path = shard_path

    def command(self):
        return ' '.join([
            self.binary_path,
            '-p', str(self.port),
            '-C', self.config_path,
            '-m', self.shard_path + '/thdb',
            '-v', os.path.basename(self.shard_path),
            '--Config.HttpServer.Threads', '4',
        ])

    def run(self):
        container = container_lib.ContainerRunner(self.container_path)
        if container.status() == 'running' and container.get_command() == self.command():
            _log.debug('%s already started', self.container_path)
        else:
            _log.info('will run %s', self.container_path)
            container_lib.ContainerRunner(self.container_path).run(self.command())

    def stop(self):
        container_lib.ContainerRunner(self.container_path).stop()

    def status_ok(self):
        return container_lib.ContainerRunner(self.container_path).status() == 'running'

    @property
    def container_path(self):
        return 'th_{}'.format(self.port)


class Plugin(v2.plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)
        parser.add_argument('--binary-path', help='Path to binary', required=True)
        parser.add_argument('--config-path', help='Path to config', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root
        self._binary_path = args.binary_path
        self._config_path = args.config_path
        self._instances = {}
        gevent.spawn(self._listen_port)

    def collect_status(self):
        return {'instances': [
            {'port': instance.port, 'status_ok': instance.status_ok()}
            for instance in self._instances.values()
        ]}

    def apply_config(self, content):
        for instance in content['instances']:
            self._instances[instance['port']] = Instance(
                self.host, instance['port'],
                binary_path=self._binary_path,
                config_path=self._config_path,
                shard_path=os.path.join(self._shard_root, instance['shard']),
            )
        for instance in self._instances.values():
            instance.run()

    def stop(self):
        for instance in self._instances.values():
            try:
                instance.stop()
            except Exception:
                _log.exception('')

    def status_ok(self):
        return all(instance.status_ok() for instance in self._instances.values())

    def _listen_port(self):
        def dummy_handler(env, start_response):
            start_response('200 OK' if self.status_ok() else '503 NOT_READY', {})
            return []
        gevent.pywsgi.WSGIServer(('::', int(self.port)), dummy_handler).serve_forever()


_log = logging.getLogger(__name__)
