import logging
import os
import requests
import retry
import tempfile

import intl2
import plugin


_REQUESTS_TIMEOUT = 30


class Plugin(plugin.BasePlugin):
    def apply_config(self, content):
        config_template = _get_template_config(self.host, self.port)

        patched_config = intl2.patch_tier1_int_config(config_template, _fix_hostnames(content))

        _save(patched_config, self.host, self.port)
        self.stop()

    def collect_status(self):
        status = {}

        current_config = _get_int_config(self.host, self.port)

        if current_config:
            status['status'] = intl2.get_shards_basesearch(current_config)

        return status

    def stop(self):
        r = requests.get(_url(self.host, self.port, 'admin', action='shutdown'), timeout=_REQUESTS_TIMEOUT)
        r.raise_for_status()


@retry.retry(tries=5, delay=1)
def _get_int_config(host, port):
    try:
        r = requests.get(_url(host, port, 'yandsearch', info='getconfig'), timeout=_REQUESTS_TIMEOUT)
        int_config = r.content
    except RuntimeError:
        logging.debug('Get config from %s:%s failed', host, port)
        int_config = None

    return int_config


def _get_template_config(host, port):
    config_template = '{}:{}.cfg.template'.format(_shortname(host), port)
    with open(config_template) as fp:
        return fp.read()


def _save(content, host, port):
    with tempfile.NamedTemporaryFile(dir='./', delete=False) as fp:
        fp.write(content)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, './{}:{}.cfg'.format(_shortname(host), port))


def _shortname(host):
    return host.split('.')[0]


def _fix_hostnames(patch):
    patch_patched = {}

    for shard, hosts in patch.items():
        patch_patched[shard] = {}
        for oldhost, newhost in hosts.items():
            patch_patched[shard][oldhost.replace('!', '.')] = newhost

    return patch_patched


def _url(host, port, path, action=None, info=None):
    url = 'http://{}:{}/{}'.format(host, port, path)
    if action:
        url = '{}?action={}'.format(url, action)
    elif info:
        url = '{}?info={}'.format(url, info)
    return url
