import json
import logging
import os
import random
import signal
import tempfile
import time

import retry
import infra.callisto.configs.client as configs
import infra.callisto.configs.config as config_format
import infra.callisto.reports as reports

import utils
import v2


class TerminateGuard(object):
    stopped = False

    def __init__(self):
        for signo in [signal.SIGINT, signal.SIGTERM]:
            signal.signal(signo, self._signal_handler)

    def _signal_handler(self, signo, _):
        logging.info('Got signal %s, going to gracefully stop', signo)
        self.stopped = True


def run_service(host, port, tags, configs_url, reports_url, mode, interval, settings):
    configs_client = configs.Client(configs_url)
    plugin = v2.MODES[mode].Plugin(host, port, tags, settings)
    plugin.start()

    timestamp = 0

    terminate_guard = TerminateGuard()
    while not terminate_guard.stopped:
        # noinspection PyBroadException
        try:
            if settings.test_config:
                config = _load_test_config(settings.test_config)
            else:
                config = _load_config(configs_client, host, port)

            if not config:
                logging.debug('No config')
            elif config.timestamp > timestamp:
                logging.info('New timestamp: [%s] -> [%s]', timestamp, config.timestamp)
                if settings.readonly:
                    logging.info('Readonly mode, not applying config')
                else:
                    _save(host, port, config.content)
                    plugin.apply_config(config.content)

                timestamp = config.timestamp

        except Exception:
            logging.exception('Unhandled exception while applying config')

        # noinspection PyBroadException
        try:
            status = plugin.collect_status()
            logging.debug('Status collected: %s', status)

            _send_report(reports_url, status, tags, host, port, utils.get_node())

        except Exception:
            logging.exception('Unhandled exception while collecting status')

        sleep_time = 5 + int(_tiny_random(interval, 1.0))
        logging.debug('sleep %s seconds', sleep_time)

        for i in range(sleep_time):
            time.sleep(1)
            if terminate_guard.stopped:
                break

    logging.info('Stopping service')
    plugin.stop()
    logging.info('Successfully stopped service')


def _send_report(reports_url, content, tags, host, port, node):
    report = reports.report.Report(host, port, tags, data=content, node=node)
    reports.client.Client(reports_url).report(report)


def _save(host, port, content):
    curdir = os.getcwd()
    with tempfile.NamedTemporaryFile(dir=curdir, delete=False) as fp:
        json.dump(content, fp, indent=4)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, os.path.join(curdir, '{}:{}.lastconfig.json'.format(utils.get_shortname(host), port)))


def _load_test_config(config_file):
    return config_format.Config(
        content=json.load(open(config_file)),
        timestamp=1
    )


@retry.retry(tries=3, delay=5, backoff=1.7)
def _load_config(configs_client, host, port):
    return configs_client.load_config(host, port)


def _tiny_random(x, max_deviation=0.05):
    centered_random = 1 - random.random() * 2  # value in (-1, +1)
    abs_max_deviation = float(max_deviation) * x
    return x + centered_random * abs_max_deviation
