import os
import logging
import tempfile

import retry
import requests

import plugin
import infra.callisto.deploy.storage.storage as resources_storage


_REQUESTS_TIMEOUT = 30


class Plugin(plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root

    def collect_status(self):
        current_shard = _get_mmeta_shard(self.host, self.port)
        return {
            current_shard: {
                'status': 'RUN',
            }
        }

    def apply_config(self, content):
        target_shard = content.get('shard')
        if not target_shard:
            raise RuntimeError('Have got no shard in config')
        self._apply_config(target_shard, content.get('resource'))

    def _apply_config(self, target_shard, target_resource=None):
        try:
            current_shard = _get_mmeta_shard(self.host, self.port)
            logging.debug('Current shard from instance: [%s]', current_shard)
            if target_shard != current_shard:
                self._prepare_config(target_shard, target_resource)
                self.stop()
        except (RuntimeError, requests.exceptions.ConnectionError) as ex:
            logging.debug('mmeta is probably not working, placing new config: [%s]', target_shard)
            self._prepare_config(target_shard, target_resource)
            raise ex

    def _prepare_config(self, target_shard, target_resource=None):
        if target_resource:
            target_shard_dir = resources_storage.resource_path(
                self._shard_root, target_resource['namespace'], target_resource['name']
            )
        else:
            target_shard_dir = os.path.join(self._shard_root, target_shard)

        if not os.path.exists(target_shard_dir):
            raise RuntimeError('Shard %s not found', target_shard_dir)

        _atomic_write('shard.conf', 'export SHARD_DIR={}'.format(target_shard_dir))

    def stop(self):
        r = requests.get(_url(self.host, self.port, 'admin', action='shutdown'), timeout=_REQUESTS_TIMEOUT)
        r.raise_for_status()


@retry.retry(exceptions=(RuntimeError, requests.exceptions.ConnectionError), tries=5, delay=1)
def _get_mmeta_shard(host, port):
    r = requests.get(_url(host, port, 'yandsearch', info='getconfig'), timeout=_REQUESTS_TIMEOUT)

    if r.ok:
        index_dir_shard, rearrange_dir_shard = None, None
        for line in r.iter_lines():
            index_dir_shard = index_dir_shard or _find_index_dir_shard(line)
            rearrange_dir_shard = rearrange_dir_shard or _find_rearr_index_dir_shard(line)
        return index_dir_shard or rearrange_dir_shard
    else:
        raise RuntimeError('get_mmeta_status failed: %s' % r.reason)


def _find_index_dir_shard(line):
    # e.g. "IndexDir path/to/shard-name/rearr"
    if ' IndexDir ' not in line:
        return None
    try:
        return _shard_name(line).split(os.path.sep)[-2].strip()
    except IndexError:
        return None


def _find_rearr_index_dir_shard(line):
    # e.g. "RearrangeIndexDir path/to/shard-name"
    if ' RearrangeIndexDir ' in line:
        line = line.partition('RearrangeIndexDir ')[2]
        return _shard_name(line).split(os.path.sep)[-1].strip()
    return None


def _atomic_write(file_name, data):
    with tempfile.NamedTemporaryFile(dir='./', delete=False) as f:
        f.write(data)
        f.close()
        os.chmod(f.name, 0644)
        os.rename(f.name, file_name)


def _url(host, port, path, action=None, info=None):
    url = 'http://{}:{}/{}'.format(host, port, path)
    if action:
        url = '{}?action={}'.format(url, action)
    elif info:
        url = '{}?info={}'.format(url, info)
    return url


def _shard_name(path_to_shard):
    if '@resource' in path_to_shard:
        path_to_shard = path_to_shard.split('/')
        path_to_shard.remove('@resource')
        return '/'.join(path_to_shard)
    return path_to_shard
