import logging
import os
import requests
import tempfile

import retry

import utils
import infra.callisto.libraries.container as container_lib


BALANCERBIN = './balancer'
BALANCERCFG = './balancer.cfg'
LOGDIR = '/logs'
PORTS_REQUIRE = 1

MAPPINGFILE = 'mapping.lua'


class Balancer(object):
    GRACEFUL_SHUTDOWN_TIMEOUT = 20  # sec
    EXIT_TIMEOUT = 20  # sec

    def __init__(self, host, port, admin_port,
                 balancer_bin=None, balancer_cfg=None,
                 log_dir=None, ports_require=PORTS_REQUIRE):
        self._host = host
        self._port = port
        self._admin_port = admin_port
        self._balancer_bin = balancer_bin or BALANCERBIN
        self._balancer_cfg = balancer_cfg or BALANCERCFG
        self._log_dir = log_dir or LOGDIR
        self._ports_require = ports_require

        name = 'balancer_{}'.format(port)
        self._container = container_lib.ContainerRunner(name)

        _write_mapping_lua({})
        self.run()

    @property
    def port(self):
        return self._port

    @property
    def admin_port(self):
        return self._admin_port

    def apply_config(self, config):
        _write_mapping_lua(config)
        try:
            self._reload_config()
        except requests.exceptions.RequestException as e:
            logging.error("Could not reload balancer config: %s", e)

    def collect_status(self):
        return {
            self._port: self._get_stats()
        }

    def run(self):
        cwd = os.getcwd()
        self._container.run(
            self._get_balancer_cmd(self._balancer_bin, self._balancer_cfg),
            env={'LUA_PATH': r'{}\;\;'.format(os.path.join(cwd, MAPPINGFILE))}
        )

    def stop(self):
        self._container.set_respawn(False)
        self._shutdown(timeout=self.GRACEFUL_SHUTDOWN_TIMEOUT)
        self._container.wait(timeout=self.EXIT_TIMEOUT)
        self._container.stop()

    def _get_balancer_cmd(self, entry_point, configfile):
        return '{entry_point} {configfile} {vars}'.format(
            entry_point=entry_point,
            configfile=configfile,
            vars=self._get_vars('cmd'),
        )

    @retry.retry(delay=5, tries=5)
    def _reload_config(self):
        requests.get(self._url(action='reload_config'), timeout=20)

    def _get_stats(self, timeout=1):
        try:
            requests.get(self._url(action='stats'), timeout=timeout)
        except requests.exceptions.RequestException:
            return False
        return True

    def _shutdown(self, timeout):
        try:
            requests.get(self._url(action='graceful_shutdown'), timeout=timeout)
        except requests.exceptions.RequestException:
            return False
        return True

    def _url(self, action):
        url = 'http://localhost:{}/{}'.format(self.admin_port, 'admin')
        if action == 'reload_config':
            timeouts = '&timeout=1s&peek_timeout=1s'
            url = '{url}?action={action}&new_config_path={cfg}{vars}'.format(
                url=url,
                action=action,
                cfg=self._balancer_cfg,
                vars=self._get_vars('url'),
            ) + timeouts
        elif action == 'graceful_shutdown':
            timeouts = '&timeout=10s&close_timeout=5s'
            url = '{url}?action={action}'.format(url=url, action=action) + timeouts
        elif action == 'stats':
            url = '{url}?action={action}'.format(url=url, action=action)
        else:
            raise RuntimeError('Balancer action unknown')

        return url

    def _get_vars(self, use='cmd'):
        balancer_vars = dict(
            host=self._host,
            port=self.port,
            admin_port=self.admin_port,
            log_dir=self._log_dir,
            ports_require=self._ports_require,
        )
        if use == 'cmd':
            return (
                '-V HOST={host} -V PORT={port} -V ADMIN_PORT={admin_port}'
                ' -V LOGDIR={log_dir} -V PORTS_REQUIRE={ports_require}'
            ).format(**balancer_vars)
        elif use == 'url':
            return (
                '&V_HOST={host}&V_PORT={port}&V_ADMIN_PORT={admin_port}'
                '&V_LOGDIR={log_dir}&V_PORTS_REQUIRE={ports_require}'
            ).format(**balancer_vars)
        raise RuntimeError('Unknown use')


def _write_mapping_lua(timestamp2port):
    """
    mapping = {
        {
            ts = 12334567;
            port = 9765;
        };
        {
            ts = 7654321;
            port = 9766;
        };
    };
    """

    with tempfile.NamedTemporaryFile(dir='.') as tmpfile:
        os.chmod(tmpfile.name, 0644)
        tmpfile.file.write('mapping = {\n')

        for timestamp, port in timestamp2port.iteritems():
            tmpfile.file.write('    {{ts = "{timestamp}";port = {port};}};\n'.format(timestamp=timestamp, port=port))

        tmpfile.file.write('};')

        utils.remove_file(MAPPINGFILE)
        os.link(tmpfile.name, MAPPINGFILE)
