import logging
import os

import requests

import infra.callisto.libraries.container as container_lib
import utils.searchconfig


class Httpsearch(object):
    GRACEFUL_SHUTDOWN_TIMEOUT = 5  # sec
    EXIT_TIMEOUT = 10  # sec
    host = None
    shard = None
    db_timestamp = None
    cache_root = None
    ports_required = 1
    log_dir = './'
    extra_arguments = None

    def __init__(self, port):
        self.port = port

        self._target = None
        self._log = logging.getLogger(self.name)
        self._container = container_lib.ContainerRunner(self.name)
        if self.cache_dir:
            utils.ensure_dir(self.cache_dir)

        self._expected_conf_hash = None

    @property
    def cache_dir(self):
        if self.cache_root:
            return os.path.join(self.cache_root, str(int(self.port)))

    @property
    def name(self):
        return 'httpsearch_{}'.format(self.port)

    def apply_config(self, config):
        self._log.info('Got new config conf_hash %s', config['conf_hash'])
        self._target = config

    @property
    def target(self):
        return self._target or {}

    def _prepare_config(self, config_template):
        return config_template

    def run(
        self,
        resources,
        upstream_pron=None, balancer_port=None, container_properties=None,
    ):
        """
        Unconditionally destroy container if exists and run new basesearch
        """
        self._shutdown(timeout=self.GRACEFUL_SHUTDOWN_TIMEOUT)
        self._expected_conf_hash = self._target['conf_hash']
        self._log.debug('Resources for run: %s', resources)

        cmd = self.get_httpsearch_cmd(
            resources,
            balancer_port or self.port,
            upstream_pron=upstream_pron,
        )

        _container_properties = {}
        _container_properties.update(container_properties or {})
        _container_properties.update(
            {
                'labels[AGENT.conf_hash]': self._target['conf_hash'],
            }
        )

        if self.extra_arguments:
            cmd += ' ' + self.extra_arguments
        self._container.run(
            cmd,
            properties=_container_properties or {}
        )

    def get_httpsearch_cmd(self, *args, **kwargs):
        raise NotImplementedError

    def collect_status(self):
        respawn_count, oom_kills = 0, 0
        if self._container.is_spawned():
            respawn_count = self._container.respawn_count
            oom_kills = self._container.oom_kills
        try:
            return {
                'shard': self.shard,
                'conf_hash': self._get_conf_hash(),
                'warmup': 100,
                'running': self._is_running(),
                'respawn_count': respawn_count,
                'oom_kills': oom_kills,
            }
        except requests.ConnectionError:
            return {
                'shard': self.shard,
                'conf_hash': self._expected_conf_hash,
                'warmup': 0,
                'running': False,
                'respawn_count': respawn_count,
                'oom_kills': oom_kills,
            }

    def stop(self):
        self._container.set_respawn(False)
        self._shutdown(timeout=self.GRACEFUL_SHUTDOWN_TIMEOUT)
        self._container.wait(timeout=self.EXIT_TIMEOUT)
        self._container.stop()

    def _get_conf_hash(self):
        return self._container.get_label('AGENT.conf_hash')

    def _is_running(self):
        return self._container.is_running

    # TODO: use timeout
    def _getconfig(self, timeout):
        r = requests.get(self._url('yandsearch', info='getconfig'), timeout=timeout)
        r.raise_for_status()

        return r

    # TODO: use timeout
    def _shutdown(self, timeout):
        try:
            requests.post(self._url('admin', action='shutdown'), timeout=timeout)
        except requests.exceptions.RequestException:
            return False
        return True

    def _url(self, path, action=None, info=None):
        if not self.host:
            raise RuntimeError('self.host is None')

        url = 'http://{}:{}/{}'.format(self.host, self.port, path)
        if action:
            url = '{}?action={}'.format(url, action)
        elif info:
            url = '{}?info={}'.format(url, info)
        return url


class Basesearch(Httpsearch):
    resources_required = (
        'basesearch.executable',
        'basesearch.cfg',
        'basesearch.models',
    )
    ports_required = 1

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['basesearch.cfg'])
        cmd = (
            '{basesearch} -p {port} -d {config}'
            ' -V MXNetFile={models} -V IndexDir={shard} -V ServerLog={log_dir}/yandex-{port}.log'
            ' -P Collection.MXNetFile={models} -P Collection.IndexDir={shard}'
            ' -P Server.ServerLog={log_dir}/yandex-{port}.log'
        ).format(
            basesearch=resources['basesearch.executable'],
            config=config,
            shard=self.shard,
            models=resources['basesearch.models'],
            port=self.port,
            log_dir=self.log_dir,
        )
        cmd += ' -V LoadLog=/dev/null -V PassageLog=/dev/null -V AbnormalityLog=/dev/null'
        cmd += ' -P Server.LoadLog=/dev/null -P Server.PassageLog=/dev/null'
        return cmd


class Embedding(Httpsearch):
    resources_required = (
        'embedding.executable',
        'embedding.models',
        'embedding.configs.server',
        'embedding.configs.storage',
        'embedding.configs.replicas',
    )
    ports_required = 1

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        server_config = self._prepare_config(
            resources['embedding.configs.server']
        )
        storage_config = self._prepare_config(
            resources['embedding.configs.storage']
        )
        replicas_config = self._prepare_config(
            resources['embedding.configs.replicas']
        )
        cmd = (
            '{executable}'
            ' --port {port}'
            ' --index-dir {shard}'
            ' --models-archive {models}'
            ' --server-config {server_config}'
            ' --embedding-storage-config {storage_config}'
            ' --embedding-storage-replics-config {replicas_config}'
            ' --db-timestamp {timestamp}'
        ).format(
            executable=resources['embedding.executable'],
            server_config=server_config,
            storage_config=storage_config,
            replicas_config=replicas_config,
            shard=self.shard,
            models=resources['embedding.models'],
            port=self.port,
            timestamp=self.db_timestamp,
        )
        return cmd


class InvertedIndex(Httpsearch):
    resources_required = (
        'invindex.executable',
        'invindex.configs.server',
        'invindex.configs.invindex',
    )
    ports_required = 1

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        server_config = self._prepare_config(
            resources['invindex.configs.server']
        )
        invindex_config = self._prepare_config(
            resources['invindex.configs.invindex']
        )
        cmd = (
            '{executable}'
            ' --port {port}'
            ' --index-dir {shard}'
            ' --server-config {server_config}'
            ' --inverted-index-config {invindex_config}'
            ' --db-timestamp {timestamp}'
        ).format(
            executable=resources['invindex.executable'],
            server_config=server_config,
            invindex_config=invindex_config,
            shard=self.shard,
            port=self.port,
            timestamp=self.db_timestamp,
        )
        return cmd


class KeyInv(Httpsearch):
    resources_required = (
        'keyinv.executable',
        'keyinv.configs.server',
        'keyinv.configs.keyinv',
    )
    ports_required = 1

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        server_config = self._prepare_config(
            resources['keyinv.configs.server']
        )
        keyinv_config = self._prepare_config(
            resources['keyinv.configs.keyinv']
        )
        cmd = (
            '{executable}'
            ' --port {port}'
            ' --server-config {server_config}'
            ' --keyinv-config {keyinv_config}'
            ' --index-dir {shard}'
        ).format(
            executable=resources['keyinv.executable'],
            server_config=server_config,
            keyinv_config=keyinv_config,
            shard=self.shard,
            port=self.port,
        )
        return cmd


class Middlesearch(Httpsearch):
    EXIT_TIMEOUT = 35

    resources_required = (
        'mmeta.executable',
        'httpsearch.cfg',
        'mmeta.models',
    )
    ports_required = 3

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['httpsearch.cfg'])
        cmd = '{executable} -p {port} -d {config}' \
              ' -V RearrangeDataDir={shard}/rearrange' \
              ' -V PureIndexDir={shard}/pure' \
              ' -V MXNetFile={models}' \
              ' -V CacheGeneration={timestamp}' \
              ' -V PlatinumTier0Timestamp={timestamp}' \
              ' -V WebTier0Timestamp={timestamp}' \
              ' -V LoadLog=/usr/local/www/logs/current-loadlog-w-mmeta-{balancer_port}' \
              ' -V EventLog=/usr/local/www/logs/current-eventlog-mmeta-{balancer_port}'.format(
                  executable=resources['mmeta.executable'],
                  config=config,
                  shard=self.shard,
                  models=resources['mmeta.models'],
                  port=self.port,
                  balancer_port=balancer_port,
                  timestamp=self.db_timestamp,
              )

        if self.cache_dir:
            cmd += ' -V QueryCacheDir={cache_dir}/search.cache'.format(cache_dir=self.cache_dir)
        if upstream_pron:
            cmd += " -V AddPron='&pipe={upstream_pron}'".format(upstream_pron=upstream_pron)

        return cmd


class MiddlesearchYP(Middlesearch):
    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['httpsearch.cfg'])
        cmd = '{executable} -p {port} -d {config}' \
              ' -V RearrangeDataDir={shard}/rearrange' \
              ' -V PureIndexDir={shard}/pure' \
              ' -V MXNetFile={models}' \
              ' -V CacheGeneration={timestamp}' \
              ' -V PlatinumTier0Timestamp={timestamp}' \
              ' -V WebTier0Timestamp={timestamp}' \
              ' -V LoadLog=/logs/current-loadlog-w-mmeta' \
              ' -V EventLog=/logs/current-eventlog-mmeta'.format(
                  executable=resources['mmeta.executable'],
                  config=config,
                  shard=self.shard,
                  models=resources['mmeta.models'],
                  port=self.port,
                  timestamp=self.db_timestamp,
              )

        if self.cache_dir:
            cmd += ' -V QueryCacheDir={cache_dir}/search.cache'.format(cache_dir=self.cache_dir)
        if upstream_pron:
            cmd += " -V AddPron='&pipe={upstream_pron}'".format(upstream_pron=upstream_pron)

        return cmd


class VideoMiddlesearch(Middlesearch):
    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['httpsearch.cfg'])
        cmd = '{executable} -p {port} -d {config}' \
              ' -V RearrangeDataDir=/db/BASE/vidmmeta_bans' \
              ' -V RearrangeIndexDir={shard}' \
              ' -V MXNetFile={models}' \
              ' -V LoadLog=/usr/local/www/logs/current-loadlog-w-mmeta-{balancer_port}' \
              ' -V EventLog=/usr/local/www/logs/current-eventlog-mmeta-{balancer_port}'.format(
                  executable=resources['mmeta.executable'],
                  config=config,
                  shard=self.shard,
                  models=resources['mmeta.models'],
                  port=self.port,
                  balancer_port=balancer_port,
              )

        if self.cache_dir:
            cmd += ' -V QueryCacheDir={cache_dir}'.format(cache_dir=self.cache_dir)
        if upstream_pron:
            cmd += " -V AddPron='&pipe={upstream_pron}'".format(upstream_pron=upstream_pron)

        return cmd


class ImgMiddlesearch(Middlesearch):
    resources_required = Middlesearch.resources_required + ('mmeta.rearrange_data', 'mmeta.rearrange_index')

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['httpsearch.cfg'])

        cmd = '{executable} -p {port} -d {config}' \
              ' -V RearrangeDataDir={rearrange_data}'  \
              ' -V RearrangeIndexDir={rearrange_index}' \
              ' -V MXNetFile={models}' \
              ' -V EventLog=/usr/local/www/logs/current-eventlog-mmeta-{balancer_port}' \
              ' -V LoadLog=/usr/local/www/logs/current-loadlog-w-mmeta-{balancer_port}' \
              ' -V PassageLog=/usr/local/www/logs/current-passagelog-base-{balancer_port}' \
              ' -V CbirTimeoutTable=150ms' \
              ' -V CbirCacheThreadLimit=8' \
              ' -V CacheGeneration={timestamp}' \
              ''.format(
                  executable=resources['mmeta.executable'],
                  config=config,
                  rearrange_data=resources.get('mmeta.rearrange_data') or '/db/BASE/imgmmeta_bans/',
                  rearrange_index=resources.get('mmeta.rearrange_index') or self.shard,
                  models=resources['mmeta.models'],
                  port=self.port,
                  balancer_port=balancer_port,
                  timestamp=self.db_timestamp,
              )

        if self.cache_dir:
            cmd += ' -V QueryCacheDir={cache_dir}'.format(cache_dir=self.cache_dir)
        if upstream_pron:
            cmd += " -V AddPron='&pipe={upstream_pron}'".format(upstream_pron=upstream_pron)

        return cmd


class RefreshMiddlesearch(Middlesearch):
    resources_required = (
        'mmeta.executable',
        'httpsearch.cfg',
        'mmeta.models',
        'mmeta.rearrange',
    )

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['httpsearch.cfg'])
        cmd = '{executable}' \
              ' -p {port}' \
              ' -d {config}' \
              ' -V RearrangeDataDir={shard}' \
              ' -V LoadLog=/usr/local/www/logs/current-loadlog-w-mmeta-{balancer_port}' \
              ' -V EventLog=/usr/local/www/logs/current-eventlog-mmeta-{balancer_port}' \
              ' -V ReAskOptions=ReAskIncompleteSources=yes,Timeout=0s,EnablePush=yes' \
              ' -V "FileCacherString=FileCacherSize 101474836480"' \
              ' -V SquidMemoryLimit=500000000' \
              ' -V SquidCacheLifeTime=5s' \
              ' -V "CacheUpdatePolicy=CacheUpdatePolicy Fasttier"' \
              ' -V EnableCheckConfig=true' \
              ' -V TimeoutTable=105ms' \
              ' -V ListingOptions=FetchMultiplier=1.3,FetchAddition=3' \
              ' -V FactorsTimeoutTable=75ms' \
              ' -V FetchTimeoutTable=55ms' \
              ' -V ReaskTimeoutTable=120ms' \
              ' -V LockRearrangeData=yes' \
              ' -V MaxSnippetsPerRequest=9' \
              ' -V MXNetFile={models}' \
              ' -V LockMXNetFile=yes'.format(
                  executable=resources['mmeta.executable'],
                  config=config,
                  shard=resources['mmeta.rearrange'],
                  models=resources['mmeta.models'],
                  port=self.port,
                  balancer_port=balancer_port,
              )

        if self.cache_dir:
            cmd += ' -V QueryCacheDir={cache_dir}'.format(cache_dir=self.cache_dir)

        return cmd


class RefreshRtyServer(Httpsearch):
    resources_required = (
        'basesearch.executable',
        'basesearch.cfg',
        'basesearch.models',
        'basesearch.static_models',
    )
    ports_required = 4
    environment = 'frozen'

    def _get_shard_name(self):
        shard_tag = utils.get_a_tag('a_shard_')
        return shard_tag.replace(
            '-0000000000',
            '-{}'.format(self._target['index_timestamp'])
        )

    def _getconfig(self, timeout=30):
        r = requests.get(self._url('', action='get_info_server'), timeout=timeout)
        r.raise_for_status()

        return r.json()

    def _shutdown(self, timeout=30):
        try:
            requests.post(self._url('', action='shutdown&rigid_level=2'), timeout=timeout)
        except requests.exceptions.RequestException:
            return False
        return True

    def _url(self, path, action=None):
        """
        /?command=get_info_server&filter=result.config
        /?command=shutdown&rigid_level=2
        """
        url = 'http://{}:{}/{}'.format(self.host, self.port + 3, path)
        if action:
            url = '{}?command={}'.format(url, action)

        self._log.info('Prepare to get URL %s', url)
        return url

    def _make_state_root(self):
        state_root_name = os.path.join(os.getcwd(), 'rtyserver-state.{}'.format(self._target['conf_hash']))
        utils.ensure_dir(state_root_name)

        return state_root_name

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        cmd = '{executable}' \
              ' {config}/rtyserver.conf-{environment}' \
              ' -E {config}/environment-{environment}' \
              ' -V CONFIG_PATH={config}' \
              ' -V CONFIGS_SUFFIX={environment}' \
              \
              ' -V LOG_POSTFIX=-{port}' \
              ' -V BasePort={port}' \
              \
              ' -V LOG_PATH=/usr/local/www/logs' \
              ' -V JournalDir=/usr/local/www/logs/' \
              ' -V STATE_ROOT={state_root}' \
              ' -V BSCONFIG_INAME=THERE_ARE_NO_BSCONFIG_VARS_HERE' \
              \
              ' -V INDEX_DIRECTORY={shard}/backup' \
              ' -P Server.IndexDir={shard}/backup' \
              ' -V DETACH_DIRECTORY={shard}/detach' \
              ' -P Server.ModulesConfig.Synchronizer.DetachPath={shard}/detach' \
              ' -V shardid={shard_id}' \
              \
              ' -V STATIC_DATA_DIRECTORY={static_models}' \
              ' -V MODELS_PATH={models_path}' \
              ' -V MXNetFile={models}'.format(
                  executable=resources['basesearch.executable'],
                  config=resources['basesearch.cfg'],
                  state_root=self._make_state_root(),
                  shard=os.path.join(self.shard, self._get_shard_name()),
                  shard_id=utils.get_a_tag('OPT_shardid='),
                  models_path=os.path.dirname(resources['basesearch.models']),
                  models=resources['basesearch.models'],
                  static_models=resources['basesearch.static_models'],
                  port=self.port,
                  environment=self.environment,
              )

        return cmd

    def _get_shard_from_config(self, cfg):
        return cfg['result']['config']['Server'][0]['IndexDir'].split(os.path.sep)[-3]

    def collect_status(self):
        status = super(RefreshRtyServer, self).collect_status()

        try:
            current_config = self._getconfig(timeout=30)
            current_shard = self._get_shard_from_config(current_config)
        except requests.ConnectionError:
            current_shard = self._get_shard_name()

        status['shard'] = current_shard

        return status


class ImgQuickRtyServer(RefreshRtyServer):
    resources_required = (
        'basesearch.executable',
        'basesearch.cfg',
        'basesearch.models',
        'basesearch.shardwriter_cfg',
        'basesearch.static_models',
    )

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        cmd = super(ImgQuickRtyServer, self).get_httpsearch_cmd(resources, balancer_port, upstream_pron)

        cmd += ' -V SHARDWRITER_CONFIG={shardwrite_config}/sw_config'.format(
            shardwrite_config=resources['basesearch.shardwriter_cfg']
        )

        return cmd


class VideoQuickRtyServer(RefreshRtyServer):
    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        cmd = super(VideoQuickRtyServer, self).get_httpsearch_cmd(resources, balancer_port, upstream_pron)

        cmd += ' -V RTINDEX_DIRECTORY=/run/shm/rtwebcache/localhost:{port}/index/project'.format(
            port=self.port,
        )

        return cmd


class NewsQuickRtyServer(RefreshRtyServer):
    environment = 'news_quick_frozen'


class IntSearch(Httpsearch):
    resources_required = (
        'int.executable',
        'int.cfg',
    )
    ports_required = 2

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['int.cfg'])
        cmd = '{executable} -p {port} -d {config}' \
              ' -V LoadLog=/usr/local/www/logs/current-loadlog-w-int-{balancer_port}' \
              ' -V EventLog=/usr/local/www/logs/current-eventlog-int-{balancer_port}'.format(
                  executable=resources['int.executable'],
                  config=config,
                  port=self.port,
                  balancer_port=balancer_port,
              )

        return cmd


class IntSearchRanking(Httpsearch):
    resources_required = (
        'int.executable',
        'int.cfg',
        'int.models',
    )
    ports_required = 2

    def get_httpsearch_cmd(self, resources, balancer_port, upstream_pron=None):
        config = self._prepare_config(resources['int.cfg'])
        cmd = '{executable} -p {port} -d {config}' \
              ' -V EventLog={log_dir}/eventlog-{port}.log' \
              ' -V LoadLog=' \
              ' -V SdLog={log_dir}/sd-{port}.log' \
              ' -V ServerLog={log_dir}/server-{port}.log' \
              ' -V MXNetFile={models}'.format(
                  executable=resources['int.executable'],
                  config=config,
                  models=resources['int.models'],
                  port=self.port,
                  log_dir=self.log_dir,
              )

        return cmd
