import logging


class Slot(object):
    def __init__(self, httpsearch_class, port, balancer_port):
        self._httpsearch_class = httpsearch_class
        self._base_port = port
        self._balancer_port = balancer_port
        self._instances = []
        self._log = logging.getLogger('slot_{}'.format(self._base_port))

    @property
    def base_port(self):
        return self._base_port

    @property
    def next_instance_port(self):
        return self._base_port + len(self._instances) * self._httpsearch_class.ports_required

    @property
    def started(self):
        hashes = set()
        for instance in self._instances:
            conf_hash = instance.target.get('conf_hash')
            if conf_hash:
                hashes.add(conf_hash)
        return hashes

    def apply_config(self, instance_config, resources, upstream_pron):
        self._log.info('Applying config %s', instance_config['conf_hash'])
        instance = self._ensure_instance(instance_config)
        instance.apply_config(instance_config)
        instance.run(
            resources,
            upstream_pron,
            self._balancer_port,
            container_properties=instance_config.get('container')
        )

    def collect_status(self):
        return [instance.collect_status() for instance in self._instances]

    @property
    def mapping(self):
        result = {}
        for instance in self._instances:
            rev = instance.target.get('revision')
            if rev:
                result[rev] = instance.port
        return result

    def reset_instances(self, instances):
        self._instances = self._filter_instances(self._instances, {instance['conf_hash'] for instance in instances})

    def _ensure_instance(self, conf):
        instance = self._find_instance(conf['revision'])
        if not instance:
            port = conf.get('port') or self.next_instance_port
            self._log.debug('Add new instance %s', conf['revision'])
            instance = self._httpsearch_class(port)
            self._instances.append(instance)
        return instance

    def _find_instance(self, revision):
        for instance in self._instances:
            if instance.target.get('revision') == revision:
                return instance

    def _filter_instances(self, instances, hashes):
        keep = []
        for instance in instances:
            if instance.target.get('conf_hash') not in hashes:
                self._log.debug('Drop instance %s', instance.target.get('conf_hash'))
                instance.stop()
            else:
                keep.append(instance)

        return keep
