import logging
import requests
import retry

import v2.plugin
import infra.callisto.libraries.container as container_lib
import infra.callisto.deploy.storage.storage as resources_storage


class Instance(object):
    def __init__(self, host, port, binary_path, config_path, shard_path):
        self.host = host
        self.port = port
        self.binary_path = binary_path
        self.config_path = config_path
        self.shard_path = shard_path

    def command(self):
        return ' '.join([
            self.binary_path,
            '--port', str(self.port),
            '--config', self.config_path,
            '--rim-db', self.shard_path,
        ])

    def run(self):
        container = container_lib.ContainerRunner(self.container_path)
        if container.is_spawned() and container.status() == 'running' and container.get_command() == self.command():
            _log.debug('%s already started', self.container_path)
        else:
            _log.info('will run %s', self.container_path)
            container_lib.ContainerRunner(self.container_path).run(self.command())

    def stop(self):
        container_lib.ContainerRunner(self.container_path).stop()

    def status(self):
        container = container_lib.ContainerRunner(self.container_path)

        if container.is_spawned() and container.status() == 'running':
            return _get_instance_shardname(self.host, self.port)

        raise RuntimeError('Container %s is not started', self.container_path)

    @property
    def container_path(self):
        return 'rim_{}'.format(self.port)


class Plugin(v2.plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)
        parser.add_argument('--binary-path', help='Path to binary', required=True)
        parser.add_argument('--config-path', help='Path to config', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root
        self._binary_path = args.binary_path
        self._config_path = args.config_path
        self._instance = None

    def collect_status(self):
        run_with_shard = None

        if self._instance:
            run_with_shard = self._instance.status()
            return {run_with_shard: {'status': 'RUN', 'warmup': 0}}
        else:
            raise RuntimeError('Could not collect status')

    def apply_config(self, content):
        namespace, name = content['resource']['namespace'], content['resource']['name']
        shard_path = resources_storage.resource_path(self._shard_root, namespace, name)

        self._instance = Instance(
            self.host, self.port,
            binary_path=self._binary_path,
            config_path=self._config_path,
            shard_path=shard_path + '/rimdb',
        )
        self._instance.run()

    def stop(self):
        try:
            self._instance.stop()
        except Exception:
            _log.exception('')


@retry.retry(delay=1, tries=3)
def _get_instance_shardname(host, port):
    r = requests.get(_url(host, port, 'shardname'), timeout=30)
    r.raise_for_status()

    return r.text.strip()


def _url(host, port, uri):
    return 'http://{}:{}/{}'.format(host, port, uri)


_log = logging.getLogger(__name__)
