import os
import json
import logging
import hashlib
import tempfile

import requests
import retry

import plugin


_REQUESTS_TIMEOUT = 30


class Plugin(plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--configs-path', help='path to configs dir', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._path_to_configs = args.configs_path

    def collect_status(self):
        try:
            running = _is_running(self.host, self.port)
        except requests.RequestException:
            running = False
        return {'running': running}

    def apply_config(self, content):
        searchmap = json.dumps(content['searchmap'], indent=4)
        _save_content(self._path_to_configs, 'searchmap.json', searchmap)
        if _prev_searchmap_hash() != _md5(searchmap):
            self.stop()
            _save_searchmap_hash(_md5(searchmap))

    def stop(self):
        logging.info('stopping')
        r = requests.post(_url(self.host, self.port, command='shutdown'), timeout=_REQUESTS_TIMEOUT)
        r.raise_for_status()


def _save_content(configs_dir, file_name, content):
    with tempfile.NamedTemporaryFile(dir=configs_dir, delete=False) as fp:
        fp.write(content)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, os.path.join(configs_dir, file_name))


@retry.retry(delay=1, tries=3)
def _is_running(host, port):
    r = requests.get(_url(host, port, command='get_status'), timeout=_REQUESTS_TIMEOUT)
    r.raise_for_status()
    return r.json()['status'] == 'Active'


def _prev_searchmap_hash():
    if os.path.exists('prev_searchmap_hash'):
        with open('prev_searchmap_hash') as f:
            return f.read()


def _save_searchmap_hash(hash_):
    _save_content('./', 'prev_searchmap_hash', hash_)


def _md5(data):
    m = hashlib.md5()
    m.update(data)
    return m.hexdigest()


INFO_PORT_SHIFT = 3


def _url(host, port, command=None):
    port = int(port) + INFO_PORT_SHIFT
    url = 'http://{}:{}'.format(host, port)
    if command:
        url = '{}?command={}'.format(url, command)
    return url
