import collections
import logging
import pprint
import json_diff


class ConfigsCollector(object):
    _debug = True

    def __init__(self, storage_, readonly=False):
        self._storage = storage_
        self._configs = {}
        self._head = None
        self._readonly = readonly

        if readonly:
            _log.info('started readonly mode')
        else:
            _log.warn('started not readonly mode')

    def update(self, configs):
        # PyCharm doesn't recognize collections.Mapping for unknown reason
        # noinspection PyUnresolvedReferences
        if isinstance(configs, collections.Mapping):
            configs = configs.iteritems()

        for agent, cfg in configs:
            try:
                host, port = agent
            except ValueError:
                host, port = agent.host, agent.port

            self._configs[host, port] = cfg

    # FIXME: deprecated
    def get(self, host, port):
        return (self._configs or {}).get((host, port))

    def release(self):
        if self._head is None:
            self._head = self._storage.load()
        if self._head != self._configs:
            if self._debug:
                _log.debug(_configs_diff_sample(self._head, self._configs))
            if self._readonly:
                _log.info('not inserted into %s: readonly', self._storage)
            else:
                self._storage.store(self._configs)
                self._head = self._configs
                self._configs = {}
                _log.debug('inserted configs to {}'.format(self._storage))
        else:
            _log.debug('no diff')

    @property
    def storage_name(self):
        return str(self._storage)


class NullCollector(object):
    """
    For configless controllers.
    Raises errors when receives non-empty config sets.
    """
    def __init__(self):
        self._log = _log.getChild('null')

    def update(self, configs):
        if configs:
            self._log.error('Unexpected update of %s configs', len(configs))
            self._log.error('Sample: %s', pprint.pformat(configs, depth=4)[:100])
            raise RuntimeError('Received non-empty config set')

    def release(self):
        pass

    @property
    def storage_name(self):
        return '__null__'


def _configs_diff_sample(old, new):
    old = {'{}:{}'.format(host, port): cfg for (host, port), cfg in old.iteritems()}
    new = {'{}:{}'.format(host, port): cfg for (host, port), cfg in new.iteritems()}
    diff = json_diff.diff(old, new)
    return pprint.pformat(diff, depth=3)[:5000]


_log = logging.getLogger('configs_collector')
