import json
import pprint
import datetime

import msgpack


class Config(object):
    version = 1

    __slots__ = ('content', 'timestamp')

    def __init__(self, content, timestamp):
        self.content = content
        self.timestamp = timestamp

    @property
    def datetime(self):
        return _timestamp_to_datetime(self.timestamp)

    def __repr__(self):
        max_len = 500
        cfg_sample = pprint.pformat(self.content, depth=3)
        if len(cfg_sample) > max_len:
            cfg_sample = cfg_sample[:max_len] + '...'
        return 'Config({})'.format(cfg_sample)


class ParseConfigError(TypeError):
    pass


class ConfigFormatError(ParseConfigError):
    pass


class UnknownConfigFormatVersion(ParseConfigError):
    pass


def to_dict(config):
    return {
        _Keys.Content: config.content,
        _Keys.Timestamp: config.timestamp,
        _Keys.Version: config.version,
    }


def from_dict(dct):
    try:
        if dct[_Keys.Version] != 1:
            raise UnknownConfigFormatVersion()
        return Config(
            dct[_Keys.Content],
            dct[_Keys.Timestamp],
        )
    except KeyError:
        raise ConfigFormatError


def dump_human_readable(config):
    """ :type config Config"""
    meta = '\n'.join((
        'Timestamp: {}'.format(config.timestamp),
        'DateTime: {}'.format(config.datetime),
        'Version: {}'.format(config.version),
    ))
    try:
        content = json.dumps(config.content, indent=4)
    except TypeError:
        content = str(config.content or '')
    return '{}\n\n{}'.format(meta, content)


def dump_msgpack(config):
    """ :type config Config"""
    return msgpack.dumps(to_dict(config))


def load_msgpack(data):
    obj = msgpack.loads(data)
    return from_dict(obj)


def dump_json(config):
    """ :type config Config"""
    return json.dumps(to_dict(config))


def load_json(data):
    obj = json.loads(data)
    return from_dict(obj)


def _timestamp_to_datetime(timestamp):
    return datetime.datetime.fromtimestamp(timestamp)


class _Keys(object):
    Content = 'content'
    Timestamp = 'timestamp'
    Version = 'version'
