import time
import logging

import flask
import infra.callisto.configs.config as config_format

import distributor


class AppContext(object):
    def __init__(self, storages, override_storage=None):
        self.distributor = distributor.ConfigsDistributor(storages, override_storage)
        self.distributor.start()


def create_app(storages, override_storage=None):
    app = flask.Flask(__name__)
    app._context = AppContext(storages, override_storage)
    while not app._context.distributor.is_ready:
        logging.info('waiting for configs to load. sleep 5 sec')
        time.sleep(5)

    app.add_url_rule('/configs/<host>/<int:port>', view_func=_api_get_configs)
    app.add_url_rule('/info/intersected_storages', view_func=_api_get_intersected_storages)
    return app


def _get_config(host, port, mime_type):
    wrapped_config = flask.current_app._context.distributor.get_config(host, port)

    if 'json' in mime_type:
        config = config_format.dump_json(wrapped_config)
    elif 'msgpack' in mime_type:
        config = config_format.dump_msgpack(wrapped_config)
    else:
        config = config_format.dump_human_readable(wrapped_config)
    return config


def _api_get_configs(host, port):
    try:
        mime_type = flask.request.accept_mimetypes.best_match(['application/msgpack', 'application/json', '*/*'])
        config = _get_config(host, port, mime_type)
        return flask.Response(config, 200, content_type=mime_type)
    except KeyError:
        return flask.Response('config not found', 404, content_type='text')


def _api_get_intersected_storages():
    lst = flask.current_app._context.distributor.intersected_storages
    return flask.jsonify({'storages': lst})
