import logging

import gevent

import infra.callisto.configs.config as config_format


class _StorageObserver(object):
    def __init__(self, storage):
        self._storage = storage
        self._configs = {}
        self._timestamp = None

    def load(self):
        head_timestamp = self._storage.head_timestamp
        if self._timestamp != head_timestamp:
            configs = self._storage.load(timestamp=head_timestamp)
            wrapped_configs = {
                (host, port): config_format.Config(config, head_timestamp)
                for (host, port), config in configs.items()
            }
            self._configs = wrapped_configs
            self._timestamp = head_timestamp
        return self._configs

    def __str__(self):
        return str(self._storage)


class ConfigsDistributor(gevent.Greenlet):
    update_rate = 5

    def __init__(self, storages, override_storage=None):
        super(ConfigsDistributor, self).__init__()
        self._storage_observers = [_StorageObserver(storage) for storage in storages]
        self._override_storage = override_storage
        self._configs = {}
        self._intersected_storages = set()
        self._active = False

    def update(self):
        self._configs = self._load_configs()

    @property
    def is_ready(self):
        return bool(self._configs)

    def _load_configs(self):
        configs = {}
        packs_of_configs = [observer.load() for observer in self._storage_observers]
        names = [str(observer) for observer in self._storage_observers]
        disjoint_packs_of_configs, disjoint_storage_names = _disjoint_dicts(packs_of_configs, names)
        for pack_of_configs in disjoint_packs_of_configs:
            configs.update(pack_of_configs)
        if self._override_storage:
            configs.update(self._override_storage.load())

        if len(packs_of_configs) != len(disjoint_packs_of_configs):
            logging.error(
                'skipped %i storages: [%s]',
                len(packs_of_configs) - len(disjoint_packs_of_configs),
                ' '.join(set(names) - set(disjoint_storage_names)),
            )
        self._intersected_storages = set(names) - set(disjoint_storage_names)
        return configs

    def get_config(self, host, port):
        return self._configs[host, port]

    @property
    def intersected_storages(self):
        return list(self._intersected_storages)

    def _run(self):
        self._active = True
        while self._active:
            try:
                self.update()
                logging.debug('updated configs')
            except Exception as ex:
                logging.exception(ex)
            finally:
                gevent.sleep(self.update_rate)

    def stop(self):
        self._active = False


def _disjoint_dicts(dicts, labels):
    """
        :returns list of dicts with pairwise disjoint keys
        if dict `a` and `b` have a key in common then we skip both `a` and `b`
    """
    disjoint_dicts = []
    disjoint_labels = []
    keys_sets = [frozenset(dct) for dct in dicts]
    for dct, keys_set, label in zip(dicts, keys_sets, labels):
        if not any(keys_set & set_ for set_ in keys_sets if set_ not in disjoint_dicts + [keys_set]):
            disjoint_dicts.append(dct)
            disjoint_labels.append(label)
    return disjoint_dicts, disjoint_labels
