#include <util/generic/algorithm.h>
#include <util/string/join.h>
#include "observer.h"

namespace NConfigsDistributor {
    template <typename TK, typename TV>
    TSet<TString> DisjointMaps(const TVector<TString>& labels, const TVector<TMap<TK, TV>>& maps);


    TCollectionObserver::TCollectionObserver(TStorage storage)
            : Storage_(std::move(storage))
    {
    }

    const TConfigsMapping& TCollectionObserver::GetConfigs() {
        auto head_timestamp = Storage_.HeadTimestamp();
        if (head_timestamp != Timestamp_) {
            auto configs = Storage_.Load(head_timestamp);
            Configs_ = std::move(configs);
            Timestamp_ = head_timestamp;
        }
        return Configs_;
    };


    TConfigsKeeper::TConfigsKeeper(TVector<NConfigsDistributor::TCollectionObserver> observers)
            : Observers_(observers)
            , Configs_()
    {
    }

    void TConfigsKeeper::Update() {
        TVector<TString> collections_names;
        TVector<TConfigsMapping> packs_of_configs;
        for (auto& observer : Observers_) {
            collections_names.push_back(observer.Name());
            packs_of_configs.push_back(observer.GetConfigs());
        }
        auto disjoint_names = DisjointMaps(collections_names, packs_of_configs);

        auto result = MakeAtomicShared<TConfigsMapping>();
        auto intersected_collections = MakeAtomicShared<TSet<TString>>();
        for (int i = 0; i < (int)collections_names.size(); ++i) {
            if (disjoint_names.count(collections_names[i]) != 0) {
                result->insert(begin(packs_of_configs[i]), end(packs_of_configs[i]));
            } else {
                intersected_collections->insert(collections_names[i]);
            }
        }
        Configs_ = result;
        Intersected_ = intersected_collections;

        Cerr << "Updated configs. Intersected [" << JoinRange(" ", begin(*Intersected_), end(*Intersected_)) << "]" << Endl;
    }

    TConfig TConfigsKeeper::GetConfig(const NConfigsDistributor::TInstance &instance) const {
        auto configs = Configs_;
        return configs->at(instance);
    }


    template<typename TK, typename TV>
    TSet<TK> ExtractKeys(TMap<TK, TV> const& input_map) {
        TSet<TK> result;
        for (auto const& element : input_map) {
            result.insert(element.first);
        }
        return result;
    }

    template <typename T>
    bool AreSetsIntersected(TSet<T> a, TSet<T> b) {
        TVector<TInstance> out;
        SetIntersection(
                begin(a), end(a),
                begin(b), end(b),
                std::inserter(out, end(out))
        );
        return !out.empty();
    }

    template <typename TK, typename TV>
    bool AreMapsIntersected(const TMap<TK, TV>& a, const TMap<TK, TV>& b) {
        return AreSetsIntersected(ExtractKeys(a), ExtractKeys(b));
    };

    template <typename TK, typename TV>
    TSet<TString> DisjointMaps(const TVector<TString>& labels, const TVector<TMap<TK, TV>>& maps) {
        assert(labels.size() == maps.size());
        TSet<TString> disjoint_labels(begin(labels), end(labels));

        for (int i = 0; i < (int)labels.size(); ++i) {
            for (int j = 0; j < (int)labels.size(); ++j) {
                if (i != j && AreMapsIntersected(maps[i], maps[j])) {
                    disjoint_labels.erase(labels[i]);
                    break;
                }
            }
        }
        return disjoint_labels;
    };
}
