import logging
import datetime

import bson
import zlib
import msgpack
import pymongo
import aniso8601
import pymongo.errors


class StorageError(RuntimeError):
    pass


class Mongo(object):
    def __init__(self, collection):
        self._coll = collection
        self._logger = logging.getLogger('MongoStorage({})'.format(collection.name))

    def store(self, id_, context):
        try:
            self._store(id_, context)
        except pymongo.errors.PyMongoError:
            raise StorageError()

    def load(self, id_, null_obj=None):
        try:
            return self._load(id_, null_obj)
        except pymongo.errors.PyMongoError:
            raise StorageError()

    def _store(self, id_, context):
        zipped_context = _binary(context)
        self._coll.update(
            {_Keys.Id: id_},
            {
                _Keys.Id: id_,
                _Keys.Context: zipped_context,
                _Keys.Timestamp: _datetime_to_timestamp(datetime.datetime.now()),
                _Keys.Zipped: True,
            },
            upsert=True
        )

    def _load(self, id_, null_obj=None):
        for rec in self._coll.find({_Keys.Id: id_}):
            if rec.get(_Keys.Zipped):
                return _de_binary(rec[_Keys.Context])
            return rec[_Keys.Context]
        return null_obj


class ROStorage(object):
    def __init__(self, storage):
        self._dict = {}
        self._storage = storage

    def store(self, id_, context):
        self._dict[id_] = context

    def load(self, id_, null_obj=None):
        res = self._dict.get(id_, null_obj)
        if res != null_obj:
            return res
        return self._storage.load(id_, null_obj)


class NullStorage(object):
    def __init__(self):
        self._dict = {}

    def store(self, id_, context):
        self._dict[id_] = context

    def load(self, id_, null_obj=None):
        return self._dict.get(id_, null_obj)


def make_storage(collection):
    collection.ensure_index([
        (_Keys.Id, 1)
    ], unique=True)
    return Mongo(collection)


class _Keys(object):
    Id = 'id'
    Context = 'c'
    Timestamp = 'ts'
    Zipped = 'z'


def _datetime_to_timestamp(dt):
    return int(dt.strftime('%s'))


def _binary(data):
    return bson.Binary(zlib.compress(msgpack.dumps(
        data,
        default=encode_datetime,
    )))


def _de_binary(data):
    return msgpack.loads(
        zlib.decompress(data),
        object_hook=decode_datetime,
    )


def decode_datetime(obj):
    if b'__datetime__' in obj:
        obj = aniso8601.parse_datetime(obj['as_str'])
    return obj


def encode_datetime(obj):
    if isinstance(obj, datetime.datetime):
        obj = {'__datetime__': True, 'as_str': obj.isoformat()}
    return obj
