import logging
import datetime
import collections

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.utils.funcs as funcs

import report as report_format
import planner
from .state import target as state_target
from .state import observed as state_observed


class CannotBuild(notify.ValueNotification):
    name = 'cannot-build'
    message_template = '{value} shards of {tier} cannot be built at the moment'
    ranges = (
        notify.Range(notify.NotifyLevels.IDLE, None, 0),
        notify.Range(notify.NotifyLevels.WARNING, 1, None),
    )


class AvailableBuilders(notify.ValueNotification):
    name = 'builders-available'


class BuildController(sdk.Controller):
    reports_alive_threshold = datetime.timedelta(minutes=10)
    path = 'ctrl'
    tags = None

    def __init__(
        self,
        name,
        tier,
        instance_provider,
        space_needed_full_build_coeff=1.3,
        space_needed_inc_build_coeff=0.95,
        assist_last_n_on_finish=0,
        remove_failed_tasks=True,
        max_builders_per_task=3,
        ignore_guarantees=False
    ):
        super(BuildController, self).__init__()
        self._name = name
        self._tier = tier
        self._target = state_target.Target({})
        self._observed = state_observed.calculate_observed_v2(self._target, {})
        self._instance_provider = instance_provider
        self._reports = {}
        self._logger = logging.getLogger(str(self))
        self.ignore_guarantees = ignore_guarantees

        self._space_needed_full = space_needed_full_build_coeff * self._shard_size
        self._space_needed_inc = space_needed_inc_build_coeff * self._shard_size
        self._assist_last_n_on_finish = assist_last_n_on_finish
        self._remove_failed_tasks_enabled = remove_failed_tasks
        self._max_builders_per_task = max_builders_per_task

        self.tags = self._instance_provider.tags

        self.add_handler('/available_builders', self._available_builders_api)

    @property
    def id(self):
        return 'build_' + self._name

    def set_context(self, context):
        self._target = _load_targets(context, frozenset(self._instance_provider.agents))

    def update(self, reports):
        reports = _filter_reports(reports, frozenset(self._instance_provider.agents))
        reports = funcs.imap_ignoring_exceptions(report_format.convert_report_to_builder_report_v2, reports)
        self._reports = {rep.agent: rep for rep in reports}
        self._observed = state_observed.calculate_observed_v2(self._target, self._reports)

    def execute(self):
        self._mark_done_tasks()
        self._assign_tasks()
        self._remove_failed_tasks()
        self._remove_old_excessive_tasks()
        self._assist_on_finish()

        cannot_build_shards = len(self._tasks_to_build())
        if cannot_build_shards > 0:
            self._logger.warning('%i shards cannot be built at the moment', cannot_build_shards)

    def gencfg(self):
        configs = {
            (agent.host, agent.port): {'shards': {}}
            for agent in self._instance_provider.agents
        }
        for target in self._target.task_targets():
            for agent in target.all_agents:
                configs[agent.host, agent.port]['shards'][target.task.resource_name] = target.task.produce_config()
        return configs

    def get_context(self):
        return self._target.json()

    def build(self, task):
        self._target.build(task)

    def keep(self, task_id):
        self._target.keep(task_id)

    def remove(self, task_id):
        self._target.remove(task_id)

    def notifications(self):
        lst = list()
        lst.append(AvailableBuilders(
            len(self._available_builders()),
            labels=dict(tier=self._tier.name)
        ))
        if len(tuple(self._instance_provider.agents)) >= self._tier.shards_count:
            lst.append(CannotBuild(
                value=len(self._tasks_to_build()),
                tier=self._tier,
                labels=dict(tier=self._tier.name),
            ))
        return lst

    def _remove_failed_tasks(self):
        if not self._remove_failed_tasks_enabled:
            return
        for task_id, target in self._target.iteritems():
            for agent in self._observed[task_id].failed:
                self._logger.warning('task %s failed on %s', task_id, agent)
                target.ensure_remove(agent)

    def _mark_done_tasks(self):
        for task_id, target in self._target.iteritems():
            for agent in self._observed[task_id].prepared:
                if agent in target.building:
                    target.ensure_keep(agent)

    def _plan_and_assign(self, tasks, builders):
        mapping = planner.assign_build_tasks_to_builders(
            tasks_to_build=tasks,
            builders=builders,
            space_needed_inc=self._space_needed_inc,
            space_needed_full=self._space_needed_full,
            shard_size=self._shard_size,
        )

        self._target.assign_tasks([
            (agent, task.task_id)
            for agent, task in mapping.iteritems()
        ])

    def _assign_tasks(self):
        self._plan_and_assign(self._tasks_to_build(), self._available_builders())

    def _assist_on_finish(self):
        self._plan_and_assign(self._tasks_to_assist_on_finish(), self._available_builders())

    def _remove_old_excessive_tasks(self):
        # HACK: if not enough space, try keep_replicas_count=0
        mapping = planner.find_tasks_to_remove(
            tasks_to_build=self._tasks_with_mode(state_target.Mode.Build),
            tasks_to_keep=self._tasks_with_mode(state_target.Mode.Keep),
            builders=self._all_builders(),
            keep_replicas_count=1,
        )

        for agent, tasks in mapping.iteritems():
            for task in tasks:
                self._target[task.task_id].ensure_remove(agent)

    def _tasks_with_mode(self, mode):
        return [
            target.task
            for target in self._target.task_targets()
            if target.mode == mode
        ]

    def _tasks_to_build(self):
        return [
            target_.task
            for task_id, target_ in self._target.iteritems()
            if target_.mode == state_target.Mode.Build
            and target_.since_last_modified > self.reports_alive_threshold
            and (not self._observed[task_id].prepared and not self._observed[task_id].building)
            and len(target_.all_agents) < self._max_builders_per_task
        ]

    def _tasks_to_assist_on_finish(self):
        tasks = [
            target_.task
            for task_id, target_ in self._target.iteritems()
            if target_.mode == state_target.Mode.Build
            and target_.since_last_modified > self.reports_alive_threshold
            and not self._observed[task_id].prepared
            and len(target_.all_agents) < self._max_builders_per_task
        ]
        if len(tasks) <= self._assist_last_n_on_finish:
            return tasks
        return []

    def _busy_agents(self):
        busy = set()
        for target in self._target.task_targets():
            if target.mode == state_target.Mode.Build:
                busy |= target.building
        return busy

    def _available_agents(self):
        return set(self._reports) - self._busy_agents()

    def _list_builders(self, agents):
        prepared_shards = self._target.shards_on_agents(agents, prepared=True)
        building_shards = self._target.shards_on_agents(agents, building=True)

        return [
            planner.Builder(
                agent=agent,
                space=self._reports[agent].freespace if self.ignore_guarantees else self._instance_provider.agents_instances[agent].storage_size,
                building_shards=building_shards[agent],
                prepared_shards=prepared_shards[agent],
                tier=self._tier
            )
            for agent in agents
        ]

    def _available_builders(self):
        return self._list_builders(self._available_agents())

    def _all_builders(self):
        return self._list_builders(set(self._reports))

    @property
    def _shard_size(self):
        return self._tier.shard_size

    def __str__(self):
        return 'BuildController({})'.format(self._name)

    def _available_builders_api(self):
        return [
            builder.__dict__
            for builder in self._available_builders()
        ]

    # compatibility methods

    def tasks_state(self):
        res = {}
        for task_id, target in self._target.iteritems():
            res[task_id] = _TaskState(target.task, target.mode, self._observed[task_id])
        return res


def _load_targets(context, known_agents):
    targets = state_target.Target.from_json(context) if context else state_target.Target({})
    for task_target in targets.task_targets():
        for agent in task_target.all_agents - known_agents:
            task_target.ensure_remove(agent)
    return targets


def _filter_reports(reports, known_agents):
    return (rep for agent, rep in reports.iteritems() if agent in known_agents)


_TaskState = collections.namedtuple('_TaskState', [
    'task',
    'mode',
    'observed',
])
