import collections

from ..utils import entities
from ..utils import const_obj
from ..utils import funcs


class Report(const_obj.ConstObj):
    def __init__(self, host, port, prepared, building, idle, failed, modified_at, la, freespace):
        self.host = host
        self.port = port
        self.prepared = prepared
        self.building = building
        self.idle = idle
        self.failed = failed
        self.modified_at = modified_at
        self.la = la
        self.freespace = freespace

        self.on_initialize()

    @property
    def instance(self):
        return '{}:{}'.format(self.host, self.port)

    @property
    def agent(self):
        return entities.Agent(self.host, self.port)

    @funcs.cached_property
    def all_tasks(self):
        return funcs.merge_dicts(self.prepared, self.building, self.idle, self.failed)

    @property
    def ready(self):
        return not self.building

    @property
    def busy(self):
        return not self.ready

    def __str__(self):
        return 'BuildAgent(%s, %s)' % (self.host, self.port)


_Task = collections.namedtuple('_Task', [
    'resource_name',
    'attempt',
    'fails_cnt',
    'method',
])


def convert_report_to_builder_report_v2(report):
    tasks = {
        entities.ShardStatus.Prepared: dict(),
        entities.ShardStatus.Building: dict(),
        entities.ShardStatus.NotStarted: dict(),
        entities.ShardStatus.Failed: dict(),
    }

    for name, state in report.data.get('shards', {}).items():
        tasks[state['status']][name] = _Task(
            resource_name=name,
            attempt=state['attempt'],
            fails_cnt=max(0, state['attempt'] - 1),
            method=state['method'],
        )

    assert report.data['la'] is not None
    assert report.data['freespace'] is not None

    return Report(
        report.host,
        report.port,
        tasks[entities.ShardStatus.Prepared],
        tasks[entities.ShardStatus.Building],
        tasks[entities.ShardStatus.NotStarted],
        tasks[entities.ShardStatus.Failed],
        report.generation_time,
        report.data['la'],
        report.data['freespace']
    )
