import datetime
import collections

import retry
import yt.wrapper as yt

import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.funcs as funcs


Target = collections.namedtuple('Target', [
    'time',
    'state',
    'generation',
    'bundle_id',
    'mr_server',
    'mr_prefix',
    'prev_state',
])


class _AbstractSource(object):
    def all_targets(self):
        raise NotImplementedError()

    @classmethod
    def _leave_newest_representative(cls, targets, key_func):
        target_by_key = {}
        for target in sorted(targets, key=_sort_key):
            target_by_key[key_func(target)] = target
        return sorted(target_by_key.values(), key=_sort_key)

    def get_targets(self, unique_states=False):
        """leaves only unique generations targets"""
        if unique_states:
            return self._leave_newest_representative(self.all_targets(), lambda target: target.state)
        else:
            return self._leave_newest_representative(self.all_targets(), lambda target: target.generation)

    def target_by_generation(self, generation):
        for target in sorted(self.get_targets(), key=_sort_key, reverse=True):
            if target.generation == generation:
                return target
        raise ValueError('no targets found')

    def target_by_state(self, state):
        for target in sorted(self.get_targets(), key=_sort_key, reverse=True):
            if target.state == state:
                return target
        raise ValueError('no targets found')

    @property
    def href(self):
        return 'no-url'


class ConstSource(_AbstractSource):
    def __init__(self, targets):
        self._targets = targets

    def all_targets(self):
        return self._targets[:]


class YtTableSource(_AbstractSource):
    """
    Looks into yt tables instead of attributes.

    E.g. (//home/jupiter-test/shard_deploy/nightly_baseline)

    "Time"                      "BundleId"  "MrServer"              "MrPrefix"                      "State"
    "2017-08-01T14:11:31+0300"  312754686   "banach.yt.yandex.net"  "//home/jupiter-test/nightly"   "20170727-233232"
    "2017-08-15T20:09:53+0300"  323327367   "banach.yt.yandex.net"  "//home/jupiter-test/nightly"   "20170727-233232"
    "2017-08-21T12:59:30+0300"  325179603   "banach.yt.yandex.net"  "//home/jupiter-test/nightly"   "20170812-021024"
    """
    def __init__(self, root, proxy='banach', is_table_dynamic=False, use_time_as_generation=False):
        self.root = root
        self.proxy = proxy
        self._yt_access_time = datetime.datetime.min
        self._yt_client = yt_utils.get_yt_client(proxy)
        self._is_dynamic = is_table_dynamic
        self._use_time_as_generation = use_time_as_generation
        self._targets = []

    @retry.retry(exceptions=yt.common.YtError, tries=3, delay=2)
    def all_targets(self):
        now = datetime.datetime.now()
        if now - self._yt_access_time > datetime.timedelta(minutes=5):
            self._targets = self._read_table()
            self._yt_access_time = now

        return self._targets[:]

    def _read_table(self):
        if self._is_dynamic:
            rows = self._yt_client.select_rows('* from [{}]'.format(self.root))
        else:
            rows = self._yt_client.read_table(self.root, format='json')
        return rows_to_targets(rows, self._use_time_as_generation)

    @property
    def href(self):
        return yt_utils.gui_url(self.proxy, self.root)


def rows_to_targets(table_rows, use_time_as_generation):
    targets = []
    for row in table_rows:
        time = funcs.iso_time_to_timestamp(row['Time'])
        yt_state = funcs.yt_state_to_timestamp(row['State'])
        prev_state = funcs.yt_state_to_timestamp(row['PrevState']) if row.get('PrevState') else None
        generation = time if use_time_as_generation else yt_state
        targets.append(Target(
            time=time,
            state=yt_state,
            generation=generation,
            bundle_id=str(row.get('BundleId') or row['JupiterBundleId']),
            mr_server=row['MrServer'],
            mr_prefix=row['MrPrefix'],
            prev_state=prev_state,
        ))
    targets.sort(key=_sort_key)
    return targets


def _sort_key(target):
    return target.time
