import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk as sdk


_GB = 1024 ** 3

TASKS = {}


class TaskMetaClass(type):
    def __new__(mcs, name, bases, dct):
        class_ = super(TaskMetaClass, mcs).__new__(mcs, name, bases, dct)
        assert name not in TASKS
        TASKS[name] = class_
        return class_


class Task(object):
    task_name = None

    __metaclass__ = TaskMetaClass

    def __init__(self):
        assert self.task_name is not None

    def json(self):
        return {
            'task_name': self.task_name,
            'parameters': self.parameters
        }

    @classmethod
    def from_json(cls, data):
        task = TASKS.get(data['task_name'])
        if not task:
            raise ValueError()
        return task(**data['parameters'])

    @property
    def parameters(self):
        raise NotImplementedError()

    @property
    def task_id(self):
        raise NotImplementedError()

    @property
    def resource_name(self):
        raise NotImplementedError()

    def produce_config(self):
        raise NotImplementedError()

    def __str__(self):
        return self.task_id

    def __repr__(self):
        return self.task_id


class BuildShard(Task):
    bundle_rbtorrent = None
    callisto_mode = None
    entry_point = None
    gencfg_tier = None
    number = None
    yt_root = None
    yt_state = None
    bundle_type = None
    namespace = None
    tracker_url = None

    def __init__(self, number, yt_state, bundle_rbtorrent):
        super(BuildShard, self).__init__()
        self.number = number
        self.yt_state = yt_state
        self.bundle_rbtorrent = bundle_rbtorrent

    @property
    def parameters(self):
        return {
            'bundle_rbtorrent': self.bundle_rbtorrent,
            'number': self.number,
            'yt_state': self.yt_state,
        }

    @funcs.cached_property
    def task_id(self):
        return '{}-{}-{}-{}'.format(self.task_name, self.number, self.timestamp, self.bundle_rbtorrent[-4:])

    @property
    def tier_name(self):
        return self.gencfg_tier.name

    @property
    def timestamp(self):
        return funcs.yt_state_to_timestamp(self.yt_state)

    @property
    def task_name(self):
        return self.__class__.__name__

    @funcs.cached_property
    def resource_name(self):
        return self.gencfg_tier.make_shard(self.number, self.timestamp).fullname

    def produce_config(self):
        return {
            'method': 'bundle_build',
            'args': {
                'entry_point': self.entry_point,
                'env': {
                    'YT_PATH': self.yt_root,
                    'CALLISTO_MODE': self.callisto_mode or 'false',
                },
                'yt_state': self.yt_state,
                'yt_path': self.yt_root,
                'timestamp': self.timestamp,
                'number': self.number,
                'tier': self.gencfg_tier.name,
                'configure': True,
                'register': {
                    'iss': True,
                    'cajuper': bool(self.namespace),
                    'namespace': self.namespace,
                    'tracker_url': self.tracker_url,
                },
                'bundle': {
                    'rbtorrent': self.bundle_rbtorrent,
                    'type': self.bundle_type,
                },
                'extra_args': [],
            },
            'resource_name': self.resource_name,
            'task_id': self.task_id
        }


class BuildShardFromYtSource3(BuildShard):
    def __init__(self, number, yt_state, bundle_rbtorrent, yt_root, timestamp):
        super(BuildShardFromYtSource3, self).__init__(number, yt_state, bundle_rbtorrent)

        self.yt_root = yt_root
        self._timestamp = timestamp

    @property
    def parameters(self):
        return {
            'bundle_rbtorrent': self.bundle_rbtorrent,
            'number': self.number,
            'yt_state': self.yt_state,
            'yt_root': self.yt_root,
            'timestamp': self.timestamp,
        }

    @property
    def timestamp(self):
        return self._timestamp


class BuildShardIncremental(BuildShardFromYtSource3):
    disable_iss_shards = False
    enable_recursive_download = False

    def __init__(
        self,
        number,
        yt_state,
        bundle_rbtorrent,
        yt_root,
        timestamp,
        yt_proxy='banach.yt.yandex.net',
        prev_shard_name=None,
        namespace=None,
        tracker_url=None,
    ):
        super(BuildShardIncremental, self).__init__(number, yt_state, bundle_rbtorrent, yt_root, timestamp)

        self._yt_proxy = yt_proxy
        self._prev_shard_name = prev_shard_name

        self.namespace = namespace
        self.tracker_url = tracker_url

    @property
    def prev_shard_name(self):
        return self._prev_shard_name

    @property
    def yt_proxy(self):
        return self._yt_proxy

    @property
    def parameters(self):
        return {
            'bundle_rbtorrent': self.bundle_rbtorrent,
            'number': self.number,
            'yt_state': self.yt_state,
            'yt_proxy': self.yt_proxy,
            'yt_root': self.yt_root,
            'timestamp': self.timestamp,
            'prev_shard_name': self._prev_shard_name,
            'namespace': self.namespace,
            'tracker_url': self.tracker_url,
        }

    def produce_config(self):
        config = super(BuildShardIncremental, self).produce_config()
        if self._prev_shard_name:
            config['prev_shard_name'] = self._prev_shard_name
            config['method'] = 'inc_build'
            if self.namespace and self.tracker_url:
                # TODO: fix hack with resource.namespace
                shard = sdk.tier.parse_shard(self._prev_shard_name)
                config['prev_shard'] = {
                    'namespace': self.namespace.replace(str(self.timestamp), str(shard.timestamp)),
                    'name': self._prev_shard_name,
                    'tracker_url': self.tracker_url,
                    'recursive_download': self.enable_recursive_download,
                }
                # TODO: remove this hack as soon as erasure mode is enabled in 2 data centres
                if self.tier_name == 'WebTier1':
                    config['prev_shard']['regexp'] = self._prev_shard_name + '/(remote_storage/[0-9]*/[0-9]*$|local)'

        config['args']['yt_proxy'] = self.yt_proxy

        if self.disable_iss_shards:
            config['args']['configure'] = False
            config['args']['register']['iss'] = False

        return config


class BuildJupiterShard(BuildShardIncremental):
    callisto_mode = 'false'
    entry_point = 'JUPITER_BUNDLE/run.sh'
    bundle_type = 'jupiter'
    make_file = None
    split_mode = False  # splits shard to local and remote_storage
    enable_erasure_mode_for = ()  # e.g. 'arc'

    def produce_config(self):
        config = super(BuildJupiterShard, self).produce_config()
        if self.make_file:
            config['args']['make_file'] = self.make_file
        if self.split_mode:
            config['args']['extra_args'] += ['-l', '1']
            # share only local/remote_storage folders, not separate chunks
            config['args']['register']['recursive_share_depth'] = 1
        if self.enable_erasure_mode_for:
            config['args']['extra_args'] += ['-e', ' '.join(self.enable_erasure_mode_for)]
        return config
