import os

import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk.resource as resource

import bundle


class Observer(object):
    def __init__(
        self,
        tier,
        name,
        source,
        build_task_type,
        namespace=None,
        tracker_url=None,
    ):
        self._tier = tier
        self._name = name
        self._source = source
        self._build_task_type = build_task_type
        self._namespace_prefix = namespace
        self._tracker_url = tracker_url

    @property
    def tier(self):
        return self._tier

    @property
    def name(self):
        return self._name

    @property
    def namespace_prefix(self):
        return self._namespace_prefix

    @property
    def source(self):
        return self._source

    def _namespace(self, shard):
        if self._namespace_prefix:
            return resource.shard_to_resource(self._namespace_prefix, shard).namespace

    def make_task(self, shard):
        target = self._source.target_by_generation(shard.timestamp)
        bundle_rbtorrent = _mapping.get_resource_rbtorrent(target.bundle_id)

        if not bundle_rbtorrent:
            raise RuntimeError('Unable to create task')

        return self._build_task_type(
            shard.number,
            funcs.timestamp_to_yt_state(target.state),
            bundle_rbtorrent,
            target.mr_prefix,
            shard.timestamp,
            target.mr_server,
            self._find_prev_shard_name(target, shard),
            namespace=self._namespace(shard),
            tracker_url=self._tracker_url,
        )

    def get_last_generations(self, max_count):
        """ order (by ts field in table): (newest), (newest-1), ..., (oldest) """
        targets = self._source.get_targets()
        targets.sort(key=lambda x: x.time, reverse=True)
        return [target.generation for target in targets][:max_count]

    def _find_prev_shard_name(self, target, shard):
        if target.prev_state:
            return self.tier.make_shard(
                shard.number,
                self._source.target_by_state(target.prev_state).generation,
            ).fullname
        return None

    @property
    def tracker_href(self):
        return yt_utils.gui_url(
            'arnold',
            os.path.join('//home/cajuper/tracker', self._namespace_prefix.lstrip('/'))
        )


_mapping = bundle.BundleMapping()
