"""this module is used to cache json_view() and notifications() evaluation, because sometimes it is very expensive"""

import infra.callisto.controllers.sdk.notify as notify


def init_cache(ctrls):
    for ctrl in ctrls:
        _CACHE[ctrl] = _cache_factory()


def update_ctrl_cache(ctrl):
    old_cache = _CACHE[ctrl]
    new_cache = _cache_factory()

    for func in _FUNCS_TO_CACHE:
        try:
            new_cache[func.name] = func.evaluate(ctrl)
        except Exception as exc:
            if func.name != 'notifications':
                new_cache[func.name] = old_cache[func.name]
            new_cache['notifications'] += [notify.TextNotification(
                'Caught {} on {}.{}: {}'.format(type(exc), ctrl, func.name, exc.message),
                notify.NotifyLevels.ERROR
            )]

    _CACHE[ctrl] = new_cache


def notifications(ctrl):
    return _CACHE[ctrl]['notifications']


def json_view(ctrl):
    return _CACHE[ctrl]['json_view']


# private


_CACHE = {}


class _CtrlFunc(object):
    def __init__(self, func_name, default_factory):
        self.name = func_name
        self.default_factory = default_factory

    def evaluate(self, ctrl):
        return getattr(ctrl, self.name)()


_FUNCS_TO_CACHE = [
    _CtrlFunc('notifications', list),
    _CtrlFunc('json_view', lambda: None),
]


def _cache_factory():
    return {
        func.name: func.default_factory()
        for func in _FUNCS_TO_CACHE
    }
