import functools
import os
import logging
import contextlib

import gevent
import gevent.pool
import library.python.svn_version as svn_version

import infra.callisto.reports as reports
import infra.callisto.configs.collector as configs_collector
import infra.callisto.context.storage as context_storage
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.utils.entities as entities
import cache
import loop
import statistics
import http.path
import http.app as app
import http.context as context
import http.response as response

import infra.callisto.libraries.discovery as discovery


class SinceLastSuccessAlert(notify.ValueNotification):
    name = 'since-success'
    message_template = 'Last successful loop iteration was {value} seconds ago'
    ranges = (
        notify.Range(notify.NotifyLevels.IDLE, None, 30 * 60),
        notify.Range(notify.NotifyLevels.ERROR, 30 * 60, None),
    )


class Application(app.Api):
    def __init__(self, ctrl, configs=None, loop_statistics=None, readonly=True):
        super(Application, self).__init__(ctrl)
        self._configs = configs
        self._configs_monitor = ConfigsDistributorMonitor()
        self._loop_statistics = loop_statistics
        self._readonly = readonly

        gevent.spawn(self._configs_monitor.update)

    def _since_success_warning(self):
        data = self._loop_statistics.json()
        since_started, since_success = data['seconds_since_started'], data['seconds_since_success']
        if since_success or since_started > 30 * 60:
            return SinceLastSuccessAlert(since_success or since_started)

    def _configs_intersection_warning(self):
        if self._configs.storage_name in self._configs_monitor.intersected_storages:
            return notify.TextNotification(
                'Configs intersection ({}). Configs cannot be applied'.format(self._configs.storage_name),
                notify.NotifyLevels.ERROR,
            )

    def notifications(self):
        return [
            self._since_success_warning(),
            self._configs_intersection_warning()
        ] + super(Application, self).notifications()

    def render_viewer(self, viewer_name, json_data=None, status_code=200):
        data = dict(
            json_data=json_data,
            banners=self.render_banners(),
            readonly=self._readonly,
            svn_revision=svn_version.svn_revision(),
            root_path=http.path.root_path(),
        )
        if context.current_request().args.get('viewer-json') in ('da', 'yes', '1'):
            return self.jsonify(data)

        return response.render_viewer(
            viewer_name=viewer_name,
            data=data,
            status_code=status_code,
        )

    @app.route('/info/notifications')
    def _get_notifications(self):
        return self.jsonify([x.json() for x in self.get_notifications(notify.NotifyLevels.IDLE)])

    @app.route('/info/notifications/solomon')
    def _get_notifications_solomon(self):
        return self.jsonify({'sensors': [
            x.solomon()
            for x in self.get_notifications(notify.NotifyLevels.IDLE, solomon_only=True)
        ]})

    @app.route('/status')
    def _status(self):
        return self.jsonify(cache.json_view(self._ctrl))

    @app.route('/unistat')
    def _unistat(self):
        if not hasattr(self._ctrl, 'unistat'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.unistat())

    @app.route('/view/deploy_progress')
    @app.route('/view/deploy_progress/<tier_name>')
    def _deploy_progress(self, tier_name=None):
        if not hasattr(self._ctrl, 'deploy_progress'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.deploy_progress(tier_name))

    @app.route('/view/deploy_percentage')
    @app.route('/view/deploy_percentage/<int:timestamp>')
    def _deploy_percentage(self, timestamp=None):
        if not hasattr(self._ctrl, 'deploy_percentage'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.deploy_percentage(timestamp))

    @app.route('/view/deploy_namespace_percentage')
    def _deploy_namespace_percentage(self):
        if not hasattr(self._ctrl, 'deploy_namespace_percentage'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.deploy_namespace_percentage())

    @app.route('/view/build_progress')
    @app.route('/view/build_progress/<tier_name>')
    def _build_progress(self, tier_name=None):
        if not hasattr(self._ctrl, 'build_progress'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.build_progress(tier_name))

    @app.route('/view/searchers_state')
    @app.route('/view/searchers_state/<slot_name>')
    def _searchers_state(self, slot_name=None):
        if not hasattr(self._ctrl, 'searchers_state'):
            return self.text_response('api not implemented', status_code=400)
        return self.jsonify(self._ctrl.searchers_state(slot_name))

    @app.route('/configs/<host>/<int:port>')
    def _get_configs(self, host, port):
        config = self._configs.get(host, port)
        if config:
            return self.jsonify(config)
        else:
            return self.text_response('no config', status_code=400)

    @app.route('/info/loop_statistics')
    def _get_loop_statistics(self):
        return self.jsonify(self._loop_statistics.json())

    @app.route('/timeline')
    def _timeline_viewer(self):
        return self.render_viewer('timeline', self._loop_statistics.extended_json())

    @app.route('/info/timeline')
    def _get_timeline(self):
        return self.jsonify(self._loop_statistics.extended_json())


class ConfigsDistributorMonitor(object):
    """determines if config of any instance (host, port) exists in multiple configs collections"""
    _url = 'http://{}-configs-cajuper.n.yandex-team.ru'
    # _locations = ('man', 'sas', 'vla')
    _locations = ('sas', 'vla')

    def __init__(self):
        super(ConfigsDistributorMonitor, self).__init__()
        self._intersected_storages = {loc: set() for loc in self._locations}

    def update(self):
        import requests
        while True:
            try:
                for loc in self._locations:
                    url = self._url.format(loc) + '/info/intersected_storages'
                    intersected = requests.get(url, timeout=30).json()['storages']
                    self._intersected_storages[loc] = set(intersected)
            except Exception as exc:
                logging.debug('error in ConfigsDistributorMonitor: %s', exc)
            finally:
                gevent.sleep(120)

    @property
    def intersected_storages(self):
        return set.union(*self._intersected_storages.values())


class ReportsKeeper(object):
    def __init__(self, backends):
        self._backends = backends
        self._map = gevent.pool.Pool().imap_unordered if len(self._backends) > 1 else map

    def _iter_reports(self, tags):
        for resp in self._map(lambda backend: backend.with_tags(tags), self._backends):
            for report in resp:
                yield report

    def with_tags(self, tags):
        reports_ = {}
        for report in self._iter_reports(tags):
            agent = entities.Agent(report.host, report.port, node_name=report.node)
            if (
                agent not in reports_
                or reports_[agent].generation_time < report.generation_time
            ):
                reports_[agent] = report
        return reports_

    def drop_cache(self):
        for backend in self._backends:
            if hasattr(backend, 'drop_cache'):
                backend.drop_cache()


def make_caching_reports_keeper(backends, min_uptime):
    return ReportsKeeper([
        reports.client.CachingClient(url, min_uptime=min_uptime) for url in backends
    ])


def create_controller(make_ctrl_func, readonly, config):
    if not readonly:
        logging.error('*** ATTENTION ***')
        logging.error('rw mode enabled')
        logging.warning('last chance to stop')
        timeout = 5
        for i in range(timeout):
            logging.warning('... %s', timeout - i - 1)
            gevent.sleep(1)
    else:
        logging.info('Readonly mode')

    if config:
        return make_ctrl_func(readonly, config)
    else:
        return make_ctrl_func(readonly)


class Register(object):
    proxy = 'locke'
    base_path = '//home/cajuper/discovery'

    def __init__(self, host, port, root_ctrl, readonly):
        self._host = host
        self._port = port
        self._root_ctrl = root_ctrl.strip('/')
        self._readonly = readonly

    @property
    def path(self):
        return os.path.join(self.base_path, self._root_ctrl)

    @contextlib.contextmanager
    def _fake_lock(self):
        logging.info('lock disabled, do not lock %s', self.path)
        yield

    def lock(self):
        if not self._readonly:
            return discovery.lock_path(self.proxy, self.path)
        else:
            return self._fake_lock()

    def register(self):
        if not self._readonly:
            discovery.update_path_specs(self.proxy, self.path, host=self._host, port=self._port)
        else:
            logging.debug('lock disabled, do not register %s:%s', self._host, self._port)


def create_app(root, readonly, host, port, enable_lock, config_file):
    registry_ = registry.ROOT_CTRLS[root]

    if config_file:
        if not registry_.config_type:
            raise RuntimeError('Config type not set')
        import google.protobuf.text_format as text_format
        with open(config_file) as f:
            config = registry_.config_type()
            text_format.Parse(f.read(), config)
    else:
        config = registry_.config_type() if registry_.config_type else None

    root_controller = create_controller(registry_.make_ctrl, readonly, config)

    if not root_controller.path:
        root_controller.path = root

    reports_keeper = make_caching_reports_keeper(registry_.reports_backends or [], 300)
    if registry_.configs_storage:
        configs_keeper = configs_collector.ConfigsCollector(registry_.configs_storage.get_storage(), readonly)
    else:
        configs_keeper = configs_collector.NullCollector()

    if registry_.context_storage:
        context_ = registry_.context_storage.get_storage(readonly)
    else:
        context_ = context_storage.NullStorage()
    register = Register(host, port, root, readonly=not enable_lock)

    loop_statistics = statistics.LoopStatistics()
    return functools.partial(
        loop.loop,
        root_controllers=[root_controller],
        reports=reports_keeper,
        configs=configs_keeper,
        context=context_,
        loop_statistics=loop_statistics,
        register=register,
        sleep_time=registry_.sleep_time,
    ),  Application(
        root_controller,
        configs=configs_keeper,
        loop_statistics=loop_statistics,
        readonly=readonly,
    )
