import datetime
import collections

import requests
from werkzeug import routing
from werkzeug import wrappers

import response
import context
import viewer
import path as paths

import infra.callisto.controllers.core.links as links
import infra.callisto.controllers.core.cache as cache
import infra.callisto.controllers.sdk.blocks as blocks
import infra.callisto.controllers.sdk.notify as notify


_Route = collections.namedtuple('_Route', ['path', 'methods'])


def route(path, methods=('GET',)):
    def f(endpoint):
        routes = getattr(endpoint, 'routes', set())
        routes.add(_Route(path, tuple(methods)))
        setattr(endpoint, 'routes', routes)
        return endpoint

    return f


class BaseApp(object):
    def __init__(self, root_ctrl):
        self._ctrl = root_ctrl
        self._url_map = _make_route_map(root_ctrl, self.jsonify, self.render_viewer)
        self._add_decorated_routes()

    def __call__(self, environ, start_response):
        request = wrappers.Request(environ)
        urls = self._url_map.bind_to_environ(environ)

        with context.set_request_context(request, self._ctrl):
            return urls.dispatch(
                lambda endpoint, kwargs: endpoint(**kwargs),
                catch_http_exceptions=True
            )(environ, start_response)

    def add_rule(self, path, endpoint, methods=('GET',)):
        """ endpoint must accept wrappers.Request as first parameter """
        self._url_map.add(routing.Rule(path, endpoint=endpoint, methods=methods))

    @classmethod
    def jsonify(cls, json_data, status_code=200):
        return response.jsonify(json_data, status_code)

    @classmethod
    def render_viewer(cls, viewer_name, json_data=None, status_code=200):
        return response.render_viewer(
            viewer_name=viewer_name,
            data=dict(json_data=json_data),
            status_code=status_code,
        )

    @classmethod
    def text_response(cls, text, status_code=200):
        return response.text_response(text, status_code)

    def _add_decorated_routes(self):
        for field_name in dir(self):
            if hasattr(getattr(self, field_name), 'routes'):
                handler = getattr(self, field_name)
                routes = getattr(handler, 'routes')
                for route_ in routes:
                    self.add_rule(route_.path, handler, route_.methods)


class StaticFilesMixin(BaseApp):
    @route('/_static_/<path:path>')
    def _static_file(self, path):
        data, mime_type = viewer.get_static_file(path)
        return wrappers.Response(
            data,
            mimetype=mime_type,
        )


class LogsViewerMixin(BaseApp):
    @route('/logs')
    def _logs(self):
        builder = context.current_request().args.get('builder')
        shard = context.current_request().args.get('shard')
        if not (builder and shard):
            return wrappers.Response('pass &builder and &shard', 400)

        logs_url = 'http://{}/listdir?shard={}&attempts_cnt={}'.format(builder, shard, 20)
        logs_files = {
            f['name']: {
                'size': f['size'],
                'mtime': datetime.datetime.fromtimestamp(int(f['mtime'])).strftime('%m-%d %H:%M')
            }
            for f in requests.get(logs_url, timeout=30).json()
        }

        return self.render_viewer('logs', {
            'logs_files': logs_files,
            'shard': shard,
            'builder': builder,
        })

    @route('/logs_download')
    def _logs_download(self):
        args = context.current_request().args
        url = 'http://{}/getcontent?shard={}&file={}'.format(args['builder'], args['shard'], args['file'])
        return wrappers.Response(requests.get(url, timeout=30).text, mimetype="text/plain")


class NotificationsMixin(BaseApp):
    def notifications(self):
        result = []
        for ctrl in links.topological_sort(self._ctrl, reverse=True):
            for notification in cache.notifications(ctrl):
                result.append(notification)
        return result

    def get_notifications(self, level=notify.NotifyLevels.INFO, solomon_only=False):
        return [
            x for x in self.notifications()
            if x and x.level >= level and (not solomon_only or hasattr(x, 'solomon'))
        ]

    def render_banners(self):
        try:
            banners = [banner.as_banner() for banner in self.get_notifications()]
        except (KeyError, RuntimeError, ValueError, TypeError) as exc:
            banners = [notify.TextNotification(
                'Cannot show notifications! (catch {})'.format(exc),
                notify.NotifyLevels.ERROR
            ).as_banner()]
        return blocks.Block(banners).json()


class TreeViewerMixin(NotificationsMixin, BaseApp):
    @route('/')
    def _controllers_tree_viewer(self):
        data = {
            'tree': [_controllers_path_tree(self._ctrl)],
        }
        return self.render_viewer('tree', data)


class Api(TreeViewerMixin, LogsViewerMixin, StaticFilesMixin, BaseApp):
    pass


def _proto_handler(request, ctrl_handler):
    if hasattr(ctrl_handler, 'request_message_type'):
        message = ctrl_handler.request_message_type()
        message.ParseFromString(request.data)
        data = ctrl_handler(message)
    else:
        data = ctrl_handler()

    if request.args.get('binary') in ('da', 'yes', '1'):
        return response.text_response(data.SerializeToString(), 200, 'application/protobuf')
    else:
        return response.text_response(str(data), 200, 'text/plain')


def _path_handler(request, ctrl, jsonify, render_viewer):
    handler_path = request.args.get('handler') or request.args['path']
    if handler_path not in ctrl.handlers:
        return jsonify({'error': 'no such path'}, 404)

    ctrl_handler = ctrl.handlers[handler_path]

    if getattr(ctrl_handler, 'proto', False):
        return _proto_handler(request, ctrl_handler)

    data = _ensure_json(ctrl_handler())
    if request.args.get('viewer') in ('da', 'yes', '1'):
        return render_viewer(ctrl_handler.viewer_name, data)
    else:
        return jsonify(data)


def _view_handler(request, ctrl, jsonify, render_viewer):
    if request.args.get('viewer') in ('da', 'yes', '1'):
        return render_viewer('auto', _ensure_json(ctrl.html_view()))
    try:
        return jsonify(_ensure_json(cache.json_view(ctrl)))
    except NotImplementedError:
        return jsonify({'error': 'json_view not implemented'}, 400)


def _add_handler(route_map, path, ctrl, jsonify, render_viewer):
    def handler():
        request = context.current_request()
        if 'path' in request.args or 'handler' in request.args:
            return _path_handler(request, ctrl, jsonify, render_viewer)
        else:
            return _view_handler(request, ctrl, jsonify, render_viewer)

    route_map.add(routing.Rule(path, endpoint=handler, methods=('GET', 'POST')))


def _make_route_map(root_ctrl, jsonify, render_viewer):
    route_map = routing.Map()
    for path, ctrl in paths.get_ctrl_map(root_ctrl).items():
        _add_handler(route_map, path, ctrl, jsonify, render_viewer)
    return route_map


def _controllers_path_tree(root_ctrl):
    def visit_ctrl(root, stack=None):
        stack = (stack or []) + [root]
        node = {
            "_id": str(id(root)),
            "id": root.id,
            "name": str(root),
            "path": paths.absolute_path('/'.join([c.path for c in stack])),
            "handlers": root.handlers.keys(),
            "viewers": [key for key, func in root.handlers.items() if hasattr(func, 'viewer_name')],
            "children": [],
            "module": root.__class__.__module__,
        }
        for ctrl in root.children:
            if ctrl.path:
                node['children'].append(visit_ctrl(ctrl, stack))
        return node

    return visit_ctrl(root_ctrl)


def _ensure_json(json_data):
    if hasattr(json_data, 'json'):
        return json_data.json()
    return json_data
