import logging
import datetime

import gevent
import infra.callisto.libraries.yt as yt_utils

import links
import cache


def _set_context(context):
    for ctrl in filter(lambda x: x.id, links.all_ctrls()):
        null_obj = object()
        ctrl_context = context.load(ctrl.id, null_obj=null_obj)
        if ctrl_context != null_obj:
            ctrl.set_context(ctrl_context)


def _update(ctrl, reports, iter_started):
    if ctrl.tags is None:
        ctrl_reports = {}
    else:
        threshold = ctrl.reports_alive_threshold
        recently = (iter_started - threshold) if threshold is not None else datetime.datetime.min
        ctrl_reports = {
            agent: report
            for agent, report in reports.with_tags(ctrl.tags).items()
            if report.generation_time >= recently
        }
    ctrl.update(ctrl_reports)


def do_iteration(root_controllers, reports, configs, context):
    reports.drop_cache()
    iter_started = datetime.datetime.now()

    for root_ctrl in root_controllers:
        for ctrl in links.topological_sort(root_ctrl):
            _update(ctrl, reports, iter_started)
        for ctrl in links.topological_sort(root_ctrl, reverse=True):
            ctrl.execute()

    for ctrl in links.all_ctrls():
        configs.update(ctrl.gencfg() or {})

    for ctrl in links.all_ctrls():
        ctrl.save_status()

    for ctrl in filter(lambda x: x.id, links.all_ctrls()):
        context.store(ctrl.id, ctrl.get_context())

    configs.release()

    for ctrl in links.all_ctrls():
        cache.update_ctrl_cache(ctrl)


def _check_id_uniqueness():
    ids = set()
    for ctrl in links.all_ctrls():
        if ctrl.id is not None:
            assert ctrl.id not in ids
            ids.add(ctrl.id)


def _full_restart(
        root_controllers,
        reports,
        configs,
        context,
        loop_statistics,
        sleep_time,
        transaction
):
    _set_context(context)
    _log.info('loop started')

    while True:
        try:
            with loop_statistics.collect():
                if transaction and not transaction.is_pinger_alive():
                    raise Exception('Election lock was lost')
                do_iteration(root_controllers, reports, configs, context)
        except IOError:
            _log.exception('IOError in the loop')
        _log.info('Sleep for %s seconds', sleep_time)
        gevent.sleep(sleep_time)


def loop(
        root_controllers,
        reports,
        configs,
        context,
        loop_statistics,
        register,
        sleep_time,
):
    _check_id_uniqueness()
    cache.init_cache(links.all_ctrls())

    while True:
        try:
            with register.lock() as transaction:
                register.register()
                _full_restart(root_controllers, reports, configs, context, loop_statistics, sleep_time, transaction)
        except yt_utils.LockConflict:
            _log.debug('Unable to acquire lock')
            gevent.sleep(60)


_log = logging.getLogger(__name__)
