import datetime
import traceback
import contextlib


class Keeper(list):
    keep_last = 10

    def __init__(self):
        super(Keeper, self).__init__()
        self._counter = 0

    def push(self, item):
        del self[self.keep_last - 1:]
        self.insert(0, item)
        self._counter += 1

    @property
    def total_count(self):
        return self._counter


class Iteration(object):
    def __init__(self, started, finished):
        self.started = started
        self.finished = finished

    def json(self):
        return {
            'started': self.started.strftime('%Y-%m-%d %H:%M:%S'),
            'finished': self.finished.strftime('%Y-%m-%d %H:%M:%S'),
        }


class FailedIteration(Iteration):
    def __init__(self, started, duration, exc_traceback):
        super(FailedIteration, self).__init__(started, duration)
        exc_traceback = exc_traceback.split('\n')[3:]
        self.exc_traceback = '\n'.join(exc_traceback)

    def json(self):
        return dict(super(FailedIteration, self).json(), traceback=self.exc_traceback)


class LoopStatistics(object):
    def __init__(self):
        self._started = datetime.datetime.now()

        self._failed_iterations = Keeper()
        self._success_iterations = Keeper()

    @property
    def since_last_update(self):
        if self._success_iterations:
            return datetime.datetime.now() - self._success_iterations[0].finished
        return None

    @property
    def since_started(self):
        return datetime.datetime.now() - self._started

    def json(self):
        return {
            'seconds_since_success': self.since_last_update and self.since_last_update.total_seconds(),
            'seconds_since_started': self.since_started.total_seconds(),
            'success_iterations': self._success_iterations.total_count,
            'failed_iterations': self._failed_iterations.total_count,
        }

    def extended_json(self):
        result = self.json()
        result['succeed'] = [it.json() for it in self._success_iterations]
        result['failed'] = [it.json() for it in self._failed_iterations]
        return result

    @contextlib.contextmanager
    def collect(self):
        iter_started = datetime.datetime.now()
        try:
            yield
            iter_ = Iteration(iter_started, datetime.datetime.now())
            self._success_iterations.push(iter_)
        except Exception:
            iter_ = FailedIteration(iter_started, datetime.datetime.now(), traceback.format_exc())
            self._failed_iterations.push(iter_)
            raise
