import datetime
import collections

import report
import override

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.blocks as blocks
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.sdk.request as request
import infra.callisto.controllers.utils.disjoint_sets as disjoint_sets
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.utils.gencfg_api as gencfg_api


def make_deployer_controller(groups_and_tags, download_args_callback=None, mtn=False):
    reports_tags = set(group[0] for group in groups_and_tags)
    assert len(reports_tags) == 1, 'Different deployer groups not allowed'
    return DeployerController(
        gencfg_api.strict_host_agent_mapping(groups_and_tags, mtn=mtn),
        download_args_callback=download_args_callback,
        tags=reports_tags,
    )


def make_deployer_controller2(groups, download_args_callback=None):
    reports_tags = set(group.name for group in groups)
    assert len(reports_tags) == 1, 'Different deployer groups not allowed'
    return DeployerController(
        gencfg_api.get_strict_host_agent_mapping(groups),
        download_args_callback=download_args_callback,
        tags=reports_tags,
    )


class ShardsPrepared(notify.ValueNotification):
    name = 'shards-prepared'


class DeployerController(sdk.Controller):
    reports_alive_threshold = datetime.timedelta(minutes=10)
    path = 'deployer'

    def __init__(self, host_agent_map, download_args_callback=None, tags=None):
        super(DeployerController, self).__init__()
        self._host_agent_map = host_agent_map
        self._agents = set(host_agent_map.values())
        self._reports = {}
        self._proxies = []
        self._download_args_callback = download_args_callback or (lambda shard, host: {})
        self._tags = tags or {'a_itype_shardtool'}

        self.add_handler('/deploy_progress', self.view_progress)

    @property
    def tags(self):
        return self._tags

    def register(self, *controllers):
        super(DeployerController, self).register(*controllers)
        for proxy in controllers:
            assert hasattr(proxy, 'host_shards_target')
            if proxy not in self._proxies:
                self._proxies.append(proxy)

    def register_callback(self, callback):
        self.register(_Proxy(callback))

    def observed_on_host(self, host):
        agent = self._host_agent_map[host]
        return self._reports[agent].prepared if agent in self._reports else frozenset()

    def update(self, reports):
        reports = (report_ for agent, report_ in reports.items() if agent in self._agents)
        reports = funcs.imap_ignoring_exceptions(report.convert_report_to_deployer_report, reports)
        self._reports = {report_.agent: report_ for report_ in reports}

    def notifications(self):
        cnt = 0
        for host in self._host_agent_map:
            cnt += len(self.observed_on_host(host))
        return [ShardsPrepared(cnt)]

    @property
    def _targets(self):
        # FIXME: could be too slow
        new_targets = {agent: set() for agent in self._host_agent_map.values()}
        for proxy in self._proxies:
            for host, shards in proxy.host_shards_target().items():
                agent = self._host_agent_map[host]
                new_targets[agent] |= set(shards)
        return new_targets

    @classmethod
    def _check_null_targets(cls, targets):
        if not any(targets.values()):
            raise RuntimeError('Deploy-ctrl: empty target state')

    def _shard_cfg(self, shard, agent):
        args = self._download_args_callback(shard, agent.host)
        if shard.fullname in override.OVERRIDES:
            args['rbtorrent'] = args.get('rbtorrent', override.OVERRIDES[shard.fullname])
        return {
            'method': 'download',
            'args': args,
        }

    def gencfg(self):
        configs = {}
        targets = self._targets
        self._check_null_targets(targets)
        for agent, shards in targets.items():
            shards = {
                shard.fullname: self._shard_cfg(shard, agent)
                for shard in shards
            }
            configs[agent.host, agent.port] = {
                'shards': shards,
            }
        return configs

    def shards_state(self, tier_name=None):
        reports = self._reports
        shards_states = collections.defaultdict(_ShardState)
        for agent, shards in self._targets.iteritems():
            for shard in shards:
                if tier_name and tier_name != shard.tier.name:
                    continue
                if agent not in reports:
                    shards_states[shard].set(agent.host, _ShardState.Conditions.Dead)
                elif shard in reports[agent].prepared:
                    shards_states[shard].set(agent.host, _ShardState.Conditions.Done)
                elif shard in reports[agent].downloading:
                    shards_states[shard].set(agent.host, _ShardState.Conditions.Build)
                elif shard in reports[agent].idle:
                    shards_states[shard].set(agent.host, _ShardState.Conditions.Idle)
                else:
                    shards_states[shard].set(agent.host, _ShardState.Conditions.Idle)

        return shards_states

    def deploy_progress(self, tier_name=None):
        return {
            shard.fullname: state.json()
            for shard, state in self.shards_state(tier_name).items()
        }

    def json_view(self):
        return self.deploy_progress()

    def html_view(self):
        return blocks.Block()

    @request.add_viewer('deploy')
    def view_progress(self):
        return self.deploy_progress(request.current_request().args.get('tier_name'))


class _ShardState(disjoint_sets.EntityState):
    class Conditions(object):
        Done = 'prepared'
        Build = 'building'
        Idle = 'idle'
        Dead = 'dead'
        All = [Done, Build, Idle, Dead]

    def json(self):
        status_hosts = self.split_by_condition()
        return {
            status: status_hosts[status]
            for status in self.Conditions.All
        }


class _Proxy(sdk.Controller):
    def __init__(self, callback):
        super(_Proxy, self).__init__()
        self.callback = callback

    @property
    def path(self):
        return '__callback__'

    def host_shards_target(self):
        return self.callback()
