from collections import namedtuple

import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.sdk.tier as tier


_Deployer = namedtuple('_Deployer', [
    'host', 'port', 'prepared', 'downloading', 'idle', 'reported_at'
])


class Deployer(_Deployer):
    def __new__(cls, host, port, prepared, downloading, idle, reported_at):
        return super(cls, cls).__new__(
            cls,
            str(host),
            int(port),
            frozenset(prepared),
            frozenset(downloading),
            frozenset(idle),
            reported_at
        )

    @property
    def name(self):
        return '{}:{}'.format(self.host, self.port)

    @property
    def agent(self):
        return entities.Agent(self.host, self.port)

    @property
    def all_shards(self):
        return frozenset.union(self.prepared, self.downloading, self.idle)

    def __str__(self):
        return 'Deployer(%s, %s)' % (self.host, self.port)

    @staticmethod
    def _serialize_shards(shards):
        return [shard.fullname for shard in shards]

    def json(self):
        return dict(
            host=self.host,
            port=self.port,
            idle=self._serialize_shards(self.idle),
            downloading=self._serialize_shards(self.downloading),
            prepared=self._serialize_shards(self.prepared),
            reported_at=self.reported_at
        )


class ShardStatus:
    Prepared = 'DONE'
    Downloading = 'BUILD'
    Idle = 'IDLE'


def convert_report_to_deployer_report(report):
    shards = {ShardStatus.Prepared: set(), ShardStatus.Downloading: set(), ShardStatus.Idle: set()}
    for name, state in report.data['shards'].items():
        shards[state['status']].add(tier.parse_shard(name))

    return Deployer(
        report.host,
        report.port,
        shards[ShardStatus.Prepared],
        shards[ShardStatus.Downloading],
        shards[ShardStatus.Idle],
        report.generation_time
    )
