import datetime
import collections
import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk.request as request
import infra.callisto.controllers.utils.gencfg_api as gencfg_api

import report
import proxy as proxies


def make_controller(groups_and_tags, download_args_callback=None, mtn=False):
    reports_tags = set(group[0] for group in groups_and_tags)
    assert len(reports_tags) == 1, 'Different deployer groups not allowed'
    return DeployerController(
        gencfg_api.strict_host_agent_mapping(groups_and_tags, mtn=mtn),
        download_args_callback=download_args_callback,
        tags=reports_tags,
    )


def make_controller2(groups, download_args_callback=None, tags=None):
    reports_tags = set(group.name for group in groups)
    assert len(reports_tags) == 1, 'Different deployer groups not allowed'
    return DeployerController(
        gencfg_api.get_strict_host_agent_mapping(groups),
        download_args_callback=download_args_callback,
        tags=reports_tags,
    )


class DeployerController(sdk.Controller):
    reports_alive_threshold = datetime.timedelta(minutes=10)
    path = 'deployer'

    def __init__(self, host_agent_map, download_args_callback=None, tags=None):
        """:arg host_agent_map: agent.node_name -> agent; single instance of agent on given host"""
        super(DeployerController, self).__init__()
        self._host_agent_map = host_agent_map
        self._agents = set(host_agent_map.values())
        self._reports = {}
        self._download_args_callback = download_args_callback or (lambda resource, host: {})
        self._resources_state = {}
        self._deploy_namespace_percentage = {}
        self._tags = tags or {'a_itype_shardtool'}
        self.add_handler('/deploy_progress', self.view_progress)
        self.add_handler('/deploy_namespace_percentage', self.deploy_namespace_percentage)

    @property
    def tags(self):
        return self._tags

    def notifications(self):
        if not self._resources_state:
            return []

        result = []
        total_counts, unique_counts = _eval_counts_by_namespace(self._resources_state)
        for namespace, counts in total_counts.iteritems():
            for status in _STATUSES:
                result.append(ResourceState(value=counts[status], labels=dict(namespace=namespace, status=status)))
        for namespace, counts in unique_counts.iteritems():
            for status in _STATUSES:
                result.append(ResourceStateUnique(value=counts[status], labels=dict(namespace=namespace, status=status)))
        return result

    def register(self, *controllers):
        super(DeployerController, self).register(*controllers)
        for proxy in controllers:
            assert hasattr(proxy, 'hosts_resources_target')

    def register_callback(self, callback):
        self.register(proxies.CallbackProxy(self, callback))

    def observed_on_host(self, host, namespace_prefix):
        agent = self._host_agent_map[host]
        prepared_on_host = self._reports[agent].prepared if agent in self._reports else frozenset()
        return frozenset(
            resource
            for resource in prepared_on_host
            # TODO: how to filter nested namespace?
            # resource.namespace = /web/prod/yt, namespace_prefix /web/prod
            if resource.namespace.startswith(namespace_prefix)
        )

    def freespace_on_host(self, host):
        agent = self._host_agent_map[host]
        return self._reports[agent].freespace if agent in self._reports else 0

    def host_is_alive(self, host):
        return self._host_agent_map[host] in self._reports

    def update(self, reports):
        reports = (report_ for agent, report_ in reports.iteritems() if agent in self._agents)
        reports = funcs.imap_ignoring_exceptions(report.convert_report_to_deployer_report, reports)
        self._reports = {report_.agent: report_ for report_ in reports}

    def execute(self):
        self._resources_state = self._eval_resources_state()
        self._deploy_namespace_percentage = self._eval_deploy_namespace_percentage()

    @property
    def _targets(self):
        new_targets = {agent: set() for agent in self._host_agent_map.itervalues()}
        for proxy in self.children:
            for host, resources in proxy.hosts_resources_target().iteritems():
                skipped_resources = filter(_skip, resources)
                if skipped_resources:
                    _log.warn('skipped %s resources at %s', len(skipped_resources), host)
                    _log.warn('sample: %s', skipped_resources[0])
                agent = self._host_agent_map[host]
                new_targets[agent] |= (resources - set(skipped_resources))
        return new_targets

    def _resource_cfg(self, resource, agent):
        args = self._download_args_callback(resource, agent.host)
        args.setdefault('max_dl_speed', '30M')
        return {
            'resource': resource.to_dict(),
            'args': args,
        }

    def gencfg(self):
        configs = {}
        targets = self._targets
        _check_null_targets(targets)
        for agent, resources in targets.iteritems():
            configs[agent.host, agent.port] = {
                'resources': [
                    self._resource_cfg(resource, agent)
                    for resource in sorted(resources)
                ],
            }
        return configs

    def resources_state(self, namespace_like=None, name_like=None):
        if namespace_like or name_like:
            return {
                resource: state
                for resource, state in self._resources_state.iteritems()
                if _match_name(resource, namespace_like, name_like)
            }
        else:
            return self._resources_state

    def _eval_resources_state(self):
        reports = self._reports
        resources_state = collections.defaultdict(_resource_state)

        for agent, resources in self._targets.iteritems():
            for resource in resources:
                if agent not in reports:
                    status = 'dead'
                else:
                    status = reports[agent].resources.status(resource)
                resources_state[resource][status].add(agent.host)
        return resources_state

    def _eval_deploy_namespace_percentage(self):
        namespace_state = collections.defaultdict(lambda: {'total': [], 'done': []})
        for resource, state in self._resources_state.iteritems():
            for status in _STATUSES:
                for _ in range(len(state[status])):
                    namespace_state[resource.namespace]['total'].append(resource)
            for _ in range(len(state['prepared'])):
                namespace_state[resource.namespace]['done'].append(resource)

        return {
            namespace: {
                'resources': {
                    'percentage': float(len(set(state['done']))) / len(set(state['total'])),
                    'done': len(set(state['done'])),
                    'total': len(set(state['total'])),
                },
                'replicas': {
                    'percentage': float(len(state['done'])) / len(state['total']),
                    'done': len(state['done']),
                    'total': len(state['total']),
                },
            }
            for namespace, state in namespace_state.iteritems()
        }

    def deploy_progress(self, namespace_like=None, name_like=None):
        def _json(resource_, state_):
            return dict(state_, id=resource_.to_dict())
        return [
            _json(resource, state)
            for resource, state in self.resources_state(namespace_like, name_like).items()
        ]

    def json_view(self):
        return self.deploy_namespace_percentage()

    def html_view(self):
        return sdk.blocks.Block()

    @request.add_viewer('deploy2')
    def view_progress(self):
        args = request.current_request().args
        return self.deploy_progress(args.get('namespace'), args.get('name'))

    def deploy_namespace_percentage(self):
        return self._deploy_namespace_percentage


class ResourceState(sdk.notify.ValueNotification):
    name = 'resource-state'


class ResourceStateUnique(sdk.notify.ValueNotification):
    name = 'resource-state-unique'


def _match_name(resource, namespace_like, name_like):
    if namespace_like and namespace_like not in resource.namespace:
        return False
    if name_like and name_like not in resource.name:
        return False
    return True


def _check_null_targets(targets):
    for resources in targets.itervalues():
        if resources:
            return
    raise RuntimeError('Deploy-ctrl: empty target state')


def _resource_state():
    return {key: set() for key in _STATUSES}


def _status(resource_state):
    for status in _STATUSES:
        if resource_state[status]:
            return status


def _skip(resource):
    """:type resource: infra.callisto.deploy.resource.Resource
    e.g.
    if '1568411432' not in resource.namespace:
        return False

    return 'InvertedIndexWebTier1' in resource.name or 'GeminiTier' in resource.name
    """
    return False


_STATUSES = ['prepared', 'downloading', 'idle', 'dead']


def _eval_counts_by_namespace(resources_state):
    total_counts = collections.defaultdict(lambda: {key: int() for key in _STATUSES})
    unique_counts = collections.defaultdict(lambda: {key: int() for key in _STATUSES})

    for resource, state in resources_state.iteritems():
        _update_counts(state, total_counts[resource.namespace], unique_counts[resource.namespace])

    return total_counts, unique_counts


def _update_counts(resource_state, total_counts, unique_counts):
    for status in _STATUSES:
        total_counts[status] += len(resource_state[status])
    unique_counts[_status(resource_state)] += 1


_log = logging.getLogger(__name__)
