import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.funcs as funcs

import report


def make_controller(agents, download_params=None, tags=None, reports_alive_threshold=None):
    """
    :type agents: Iterable[infra.callisto.controllers.utils.entities.Agent]
    :type download_params: dict, e.g. {'max_dl_speed': '50M',
                                       'hardlink': True,
                                       'copier_opts': 'direct_write: 0',
                                       'check_policy': check_policy = {'repeat_sky_get': False}}
    :type tags: Iterable, e.g. {'a_itype_deployer'}
    :type reports_alive_threshold:
    """

    # don't run without speed limit
    if not download_params:
        download_params = {}
    download_params.setdefault('max_dl_speed', '30M')

    ctrl = _LightDeployerController(agents, download_params)
    if tags:
        ctrl.tags = frozenset(tags)
    if reports_alive_threshold is not None:
        ctrl.reports_alive_threshold = reports_alive_threshold
    return ctrl


class _LightDeployerController(sdk.Controller):
    """
    Healthy person's controller:
        - No http bells, no html whistles.
        - Just deploy what needs to be deployed.
        - Setup what needs to be deployed via `register_source`.
        - No public (especially http) methods in controller.
    CONSIDERED HARMFUL:
        - Check deploy status via `deployed`
    TODO: - tags
          - reports_alive_threshold
          - get rid of imap_ignoring_exceptions
          - accept dicts from deploy sources
          - push sources
          - buggy convert_report_to_deployer_report
    """
    path = 'xxx'
    tags = {}

    def __init__(self, agents, download_params):
        """
        :type agents: Iterable[infra.callisto.controllers.utils.entities.Agent]
        :type download_params: dict
        """
        super(_LightDeployerController, self).__init__()
        self._agents = frozenset(agents)
        self._agent_on_node = {agent.node_name: agent for agent in agents}
        self._report_on_node = {}
        self._download_params = download_params
        self._sources = set()  # td: Prevention of duplicate sources

    def register_source(self, source):
        """
        :param source: Callable resource source: source() returns Iterable of (node, Iterable[Resource])
        """
        assert callable(source), 'Source must be callable'
        self._sources.add(source)

    def deployed(self, node):
        # TODO: throw if node is never seen?
        return self._report_on_node[node].prepared if node in self._report_on_node else set()

    def freespace(self, node):
        return self._report_on_node[node].freespace if node in self._report_on_node else 0

    def is_alive(self, node):
        return node in self._report_on_node

    def update(self, reports):
        self._report_on_node = self._filter_reports(reports)

    def gencfg(self):
        return ((agent, {'resources': [self._resource_config(resource)
                                       for resource in sorted(resources)]})
                for agent, resources in self._compute_targets().iteritems())

    def _filter_reports(self, raw_reports):
        reports = (report_ for agent, report_ in raw_reports.iteritems() if agent in self._agents)
        reports = funcs.imap_ignoring_exceptions(report.convert_report_to_deployer_report, reports)
        return {report_.agent.node_name: report_ for report_ in reports}

    def _compute_targets(self):
        targets = {agent: set() for agent in self._agent_on_node.itervalues()}
        for source in self._sources:
            for node, resources in source():
                targets[self._agent_on_node[node]].update(resources)  # fails if pod is not scheduled
        _check_targets(targets)
        return targets

    def _resource_config(self, resource):
        return {'resource': resource.to_dict(), 'args': self._download_params}


def _check_targets(targets):
    """
    Empty target on every node is most likely an error.
    """
    for resources in targets.itervalues():
        if resources:
            return
    raise RuntimeError('Empty target state')


_log = logging.getLogger(__name__)
