import collections

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.blocks as blocks
import infra.callisto.controllers.sdk.request as request
import infra.callisto.controllers.sdk.resource as resource


class Proxy(sdk.Controller):
    def __init__(self, deploy_ctrl):
        super(Proxy, self).__init__()
        self._deploy_ctrl = deploy_ctrl
        deploy_ctrl.register(self)

    def hosts_resources_target(self):
        raise NotImplementedError()

    def observed_on_host(self, host):
        return self._deploy_ctrl.observed_on_host(host)


class CallbackProxy(Proxy):
    path = '__callback__'

    def __init__(self, deploy_ctrl, callback):
        super(CallbackProxy, self).__init__(deploy_ctrl)
        self._callback = callback

    def hosts_resources_target(self):
        return self._callback()


class ChunksProxy(Proxy):
    def __init__(self, deploy_ctrl, namespace_prefix, chunk_resources):
        super(ChunksProxy, self).__init__(deploy_ctrl)
        self._namespace_prefix = namespace_prefix
        self._stats = {}
        self._chunk_resources = chunk_resources

    def execute(self):
        self._stats = self._eval_stats()

    def html_view(self):
        return _html_view(
            stats=self._stats,
            namespace=self._namespace_prefix,
            deploy_ctrl=self._deploy_ctrl,
        )

    def json_view(self):
        return self._stats

    def host_chunks_target(self):
        raise NotImplementedError()

    def hosts_resources_target(self):
        result = {}
        for host, chunks in self.host_chunks_target().iteritems():
            result[host] = {
                resource.chunk_to_resource(self._namespace_prefix, chunk, suffix)
                for chunk in chunks
                for suffix in self._chunk_resources
            }
        return result

    def chunks_observed_on_host(self, host):
        chunks_availability = collections.defaultdict(lambda: 0)

        def converter(r):
            return resource.resource_to_chunk(self._namespace_prefix, r)
        for resource_ in self._deploy_ctrl.observed_on_host(host, self._namespace_prefix):
            try:
                chunks_availability[converter(resource_)] += 1
            except (AssertionError, ValueError):
                pass
        return {chunk for chunk, availability in chunks_availability.items() if availability == len(self._chunk_resources)}

    def _eval_stats(self):
        targets = self.host_chunks_target()
        observed = {host: self.chunks_observed_on_host(host) for host in targets}
        return _eval_stats(targets, observed)


def _split_resources_by_timestamps(targets, observed):
    timestamps = collections.defaultdict(lambda: {'done': [], 'total': []})

    for host in targets:
        for shard_or_chunk in targets[host]:
            timestamp = shard_or_chunk.timestamp
            timestamps[timestamp]['total'].append(shard_or_chunk)
            if shard_or_chunk in observed[host]:
                timestamps[timestamp]['done'].append(shard_or_chunk)

    return timestamps


def _eval_stats(targets, observed):
    result = {}
    by_timestamp = _split_resources_by_timestamps(targets, observed)
    for timestamp, stats in by_timestamp.items():
        result[timestamp] = {
            'done': len(set(stats['done'])),
            'total': len(set(stats['total'])),
            'done_replicas': len(stats['done']),
            'total_replicas': len(stats['total']),
        }
    return result


def _html_view(stats, namespace, deploy_ctrl):
    return blocks.DeployView(
        bars=[
            blocks.DoubleProgress(
                state['done'], state['total'], state['done_replicas'], state['total_replicas'],
                timestamp=timestamp
            )
            for timestamp, state in sorted(stats.items(), reverse=True)
        ],
        href_list=blocks.HrefList([
            blocks.Href(
                'deploy-progress',
                request.absolute_path(
                    request.ctrl_path(deploy_ctrl)
                ) + '?handler=/deploy_progress&namespace={}&viewer=1'.format(namespace)),
        ])
    )
