import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.const_obj as const_obj
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk as sdk


class DownloadStatus(object):
    Downloading = 'downloading'
    Prepared = 'prepared'
    Idle = 'idle'

    All = [Downloading, Prepared, Idle]


class Resources(object):
    def __init__(self, resources):
        self._status_to_resources = {
            status: frozenset(resources.get(status, frozenset()))
            for status in DownloadStatus.All
        }
        self._resource_to_status = {}
        for status, resources_ in resources.iteritems():
            for res in resources_:
                self._resource_to_status[res] = status

    def __getitem__(self, status):
        return self._status_to_resources[status]

    @funcs.cached_property
    def all(self):
        return frozenset.union(frozenset(), *self._resource_to_status.iterkeys())

    def status(self, resource_):
        return self._resource_to_status.get(resource_, DownloadStatus.Idle)


class Deployer(const_obj.ConstObj):
    def __init__(self, agent, resources, freespace, reported_at):
        self.agent = agent
        self.resources = resources
        self.freespace = freespace
        self.reported_at = reported_at

        self.on_initialize()

    @property
    def host(self):
        return self.agent.host

    @property
    def port(self):
        return self.agent.port

    @property
    def prepared(self):
        return self.resources[DownloadStatus.Prepared]

    def __str__(self):
        return 'Deployer(%s, %s)' % (self.host, self.port)


def convert_report_to_deployer_report(report):
    resources = {status: set() for status in DownloadStatus.All}
    for res in report.data['resources']:
        resources[res['status']].add(sdk.Resource(res['namespace'], res['name']))

    return Deployer(
        entities.Agent(report.host, report.port, node_name=report.node),
        Resources(resources),
        report.data['freespace'],
        report.generation_time,
    )
