# -*- coding: utf-8 -*-
"""
    Здесь и далее:
        "Slot" - группа базовых поисков, характеризуются парами инстанс-шард
        "Source" - набор слотов, на которые одновременно катим шарды и переключаем инстансы одинакового поколения
            (управляется через target табличку)
        "Contour" - набор несвязанных между собой Source'ов

    Этот код решает 2 слабо связанных задачи:
    1. Строит шарды
    2. Выкатывает шарды на группы базовых поисков

    Контур может не строить шардов вообще, а приносить на деплоеры и базовые поиски шарды, построенные где-то еще
    Может строить шарды для какого-то одного "слота", а остальные тащить откуда-то еще или whatever

    Slot'ы объединяются в SourceController,
    который деплоит на все свои слоты 2 последних поколения из своей target таблички.
    Как только поколение самой последней строки таблички приезжает на все слоты,
    SourceController переключает все базовые поиска на это поколение

    Основной контроллер просто сохраняет в status табличку target/status состояния всех своих SourceController'ов

    Таким образом, можно собрать, например такую конструкцию:

    MainYtSource = ...
    MsUserYtSource = ...
    Contour(
        deployer_groups=...,
        builders_specs=[
            BuildCtrlSpec(строим шарды Platinum из MainYtSource),
            BuildCtrlSpec(строим шарды WebTier0 из MainYtSource),
        ],
        sources_specs=[
            SourceSpec(MainYtSource, [Slot(Platinum), Slot(WebTier0)]),  # на оба слота тащим поколение из MainYtSource
            SourceSpec(MsUserYtSource, [Slot(MsUser)]),  # тащим шарды, которые строит кто-то другой
        ]                                                # поколение может не совпадать с оным из MainYtSource
    )
"""
import os

import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.deployer2.controller as deploy_controller2
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.slots.state as slots_state


class SourceController(search_source.SourceController):
    def __init__(self, name, slot_controllers, deployer, source):
        super(SourceController, self).__init__(name, slot_controllers, deployer)
        self._source = source
        self._newest_target = None

    def update(self, reports):
        targets = [
            slots_state.SlotState(target.generation)
            for target in self._source.get_targets()[-2:]
        ]
        if not targets:
            return
        self._newest_target = newest_state = targets[-1]
        deploy, search = self.get_observed_state()
        if newest_state in deploy:
            self.set_target_state(set(targets), newest_state)
        else:
            self.set_target_state(set(targets), search)

    @property
    def newest_target(self):
        return self._newest_target


class Controller(sdk.ctrl.Controller):
    path = 'MultiContour'

    def __init__(self, build_controllers, source_controllers, status_table):
        super(Controller, self).__init__()
        self._build_controllers = build_controllers
        self._source_controllers = source_controllers
        self._status_table = status_table
        self.register(*build_controllers)
        self.register(*source_controllers)

    def save_status(self):
        def _convert(slot_state):
            return str(slot_state.timestamp) if slot_state else None
        row = {}
        for source_ctrl in self._source_controllers:
            row[source_ctrl.path + 'Target'] = _convert(source_ctrl.newest_target)
            row[source_ctrl.path + 'Status'] = _convert(source_ctrl.get_observed_state()[1])
        record = self._status_table.status_class(row)
        head = self._status_table.head()
        if not head or head.status != record:
            self._status_table.write(record)

    def html_view(self):
        return sdk.blocks.wrap(
            sdk.blocks.HrefList([sdk.blocks.Href('status-table', self._status_table.gui_url)]),
            super(Controller, self).html_view(),
        )


class DeployerSpec(object):
    def __init__(self, groups, use_mtn=False):
        self.groups = groups
        self.use_mtn = use_mtn


class BuildCtrlSpec(object):
    def __init__(self, yt_observer, instance_provider, **builder_extra_args):
        self.yt_observer = yt_observer
        self.instance_provider = instance_provider
        self.builder_extra_args = builder_extra_args


class SourceSpec(object):
    def __init__(self, name, source, slots):
        self.name = name
        self.source = source
        self.slots = slots


class Contour(object):
    def __init__(self, deployer_spec, builders_specs, sources_specs, namespace_prefix):
        self.deployer_spec = deployer_spec
        self.builders_specs = builders_specs
        self.sources_specs = sources_specs
        self.namespace_prefix = namespace_prefix

        class Status(sdk.table.Status):
            schema = []
            for spec in sources_specs:
                schema += [
                    {"name": spec.name + "Target", "type": "string"},
                    {"name": spec.name + "Status", "type": "string"},
                ]

            def __init__(self, row):
                self._row = row

            @classmethod
            def load_row(cls, row):
                return cls(row)

            def dump_row(self):
                return self._row

        class StatusTable(sdk.table.StatusTable):
            status_class = Status

        self.status_class = Status
        self.status_table = StatusTable


def make_controller(contour, status_table_path, readonly):
    deploy_ctrl = deploy_controller2.make_controller(
        contour.deployer_spec.groups,
        mtn=contour.deployer_spec.use_mtn
    )
    build_controllers = [
        innerbuilder.make_controller(spec.yt_observer, spec.instance_provider, **spec.builder_extra_args)
        for spec in contour.builders_specs
    ]
    source_controllers = [
        SourceController(
            name=spec.name,
            slot_controllers=search_source.make_slot_controllers(
                slots=spec.slots,
                deploy_ctrl=deploy_ctrl,
                namespace_prefix=contour.namespace_prefix,
            ),
            deployer=deploy_ctrl,
            source=spec.source,
        )
        for spec in contour.sources_specs
    ]
    path = os.path.join('//home/cajuper/user', status_table_path.lstrip('/'))
    status_table = contour.status_table(yt_utils.create_yt_client(proxy='arnold', use_rpc=True), path, readonly)
    status_table = sdk.table.Readonly(status_table) if readonly else status_table
    return Controller(
        build_controllers=build_controllers,
        source_controllers=source_controllers,
        status_table=status_table,
    )
