import datetime

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.notify as notify


class HeadCommitNotObserved(notify.ValueNotification):
    name = 'head-commit-not-observed-seconds'


class HeadCommitNotObservedFlag(notify.ValueNotification):
    name = 'head-commit-not-observed-flag'


class Ctrl(sdk.Controller):
    @property
    def path(self):
        return self._name

    def __init__(self, name, slots_table, slots_ctrl, conf_gen, hard_switch=False):
        super(Ctrl, self).__init__()
        self._name = name
        self._slots_table = slots_table
        self._slots_ctrl = slots_ctrl
        self._known_slots = slots_ctrl.slots_ids
        self._conf_gen = conf_gen
        self._hard_switch = hard_switch
        self._head_targets = {}

        self.register(slots_ctrl, conf_gen)
        self.add_handler('/ready_commits', self._ready_commits)

    def notifications(self):
        lst = []
        for slot_id, (conf, time) in self._head_targets.items():
            if conf and conf.revision not in self._slots_ctrl.ready_revisions(slot_id):
                value = (datetime.datetime.now() - time).total_seconds()
            else:
                value = 0
            lst.append(HeadCommitNotObserved(
                value=value,
                labels=dict(slot=slot_id),
            ))
            lst.append(HeadCommitNotObservedFlag(
                value=int(bool(value)),
                labels=dict(slot=slot_id),
            ))

        return lst

    def execute(self):
        for slot_id in self._known_slots:
            configurations = self._slots_table.top_n(slot_id, 2 if not self._hard_switch else 1)
            self._head_targets[slot_id] = self._slots_table.head_target_and_time(slot_id)
            self._slots_ctrl.set_slot_target(slot_id, configurations)

    def _ready_commits(self):
        return {
            'slots': [
                {
                    'slot_id': slot_id,
                    'ready_commits': sorted(self._slots_ctrl.ready_revisions(slot_id)),
                }
                for slot_id in sorted(self._known_slots)
            ]
        }
