import datetime
import logging

import infra.callisto.controllers.sdk.request as request
import infra.callisto.controllers.sdk.blocks as blocks
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils
import infra.callisto.libraries.memoize as memoize
import infra.callisto.controllers.utils.skynet as skynet
import infra.callisto.protos.multibeta.slot_state_pb2 as slot_state_pb2

import observed
import porter


class ComponentCtrl(sdk.Controller):
    reports_alive_threshold = datetime.timedelta(minutes=10)

    is_head_slot = False  # to do new resolve api

    def __init__(self):
        super(ComponentCtrl, self).__init__()
        self._slots_targets = {}
        self._slots_default_revisions = {}
        self._observed = None

    @property
    def slots_ids(self):
        raise NotImplementedError()

    def set_target(self, slots_targets, slots_default_revisions):
        self._slots_targets = slots_targets
        self._slots_default_revisions = slots_default_revisions

    def ready_hashes(self, slot_id):
        return self._observed.ready_hashes(slot_id)

    def json(self, slot_id, conf_hash):
        data = self._observed.json(slot_id, conf_hash)
        data['self'] = str(self)
        return data

    def html(self, slot_id, conf_hash):
        return blocks.SearchView([self._observed.html(slot_id, conf_hash)], str(self))

    def _generate_configuration_config(self, instance, configuration):
        raise NotImplementedError()

    @classmethod
    def _add_extra_to_config(cls, instance, configuration):
        return {}

    def _generate_slot_config(self, instance, configurations):
        res = []
        for conf in configurations:
            try:
                cfg = self._generate_configuration_config(instance, conf)
                cfg.update(self._add_extra_to_config(instance, conf))
                res.append(cfg)
            except ResourceNotFound:
                pass
        return res

    def component_state_proto(self, slot_id, configuration):
        status = slot_state_pb2.Component.WAIT_FOR_QUORUM
        if hash(configuration) in self.ready_hashes(slot_id):
            status = slot_state_pb2.Component.READY
        return slot_state_pb2.Component(
            description=str(self),
            status=status,
            shards=self._observed.shards_progress_proto(slot_id, hash(configuration)),
            instances=self._observed.instances_progress_proto(slot_id, hash(configuration)),
            diagnostics=self._observed.diagnostics_proto(slot_id, hash(configuration)),
        )


class BaseCtrl(ComponentCtrl):
    _default_porto_properties = {
        'cpu_policy': 'idle',
        'respawn_delay': '15s',
    }

    def __init__(self, tier, agent_shard_number_map, slots_ids, porto_properties=None, tags=None):
        super(BaseCtrl, self).__init__()
        self._tier = tier
        self._agent_shard_number_map = agent_shard_number_map
        self._slots_ids = slots_ids
        self._observed = observed.BasesearchObserved(agent_shard_number_map, 0.9, 0.2, 0)

        self._porto_properties = self._default_porto_properties.copy()
        self._porto_properties.update(porto_properties or {})
        self._tags = tags or {'a_itype_base', 'a_tier_{}'.format(self._tier.name)}

    @property
    def tags(self):
        return self._tags

    @property
    def slots_ids(self):
        return self._slots_ids

    def update(self, reports):
        reports = {agent: report.data for agent, report in reports.items() if agent in self._agent_shard_number_map}
        self._observed.update(reports)

    def gencfg(self):
        configs = {}
        for agent in self._agent_shard_number_map:
            slots_config = {}
            for slot_id, configurations in self._slots_targets.items():
                if slot_id in self.slots_ids:
                    slots_config[slot_id] = {
                        'instances': self._generate_slot_config(agent, configurations),
                        'default': self._slots_default_revisions.get(slot_id),
                    }
            configs[agent] = {'slots': slots_config}

        return configs

    def _generate_configuration_config(self, agent, configuration):
        return {
            'revision': configuration.revision,
            'conf_hash': hash(configuration),
            'container': self._porto_properties,
            'resources': [
                resolve_resource('basesearch.executable', configuration.base.executable),
                resolve_resource('basesearch.models', configuration.base.models),
                self._cfg_resource(agent, configuration.base),
            ],
        }

    def _cfg_resource(self, agent, conf_bs):
        resource = resolve_resource('basesearch.cfg', conf_bs.config).copy()
        if conf_bs.config_path:
            resource['extract_file'] = conf_bs.config_path.format(
                short_host=agent.short_host, fqdn=agent.host, port=agent.port, tier=self._tier.name
            )
        return resource

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._tier)


class BaseProvidedCtrl(ComponentCtrl):
    _default_porto_properties = {
        'cpu_policy': 'idle',
        'respawn_delay': '15s',
    }

    def __init__(self, instance_provider, replication, slots_ids, porto_properties=None):
        super(BaseProvidedCtrl, self).__init__()
        self._instance_provider = instance_provider
        self._slots_ids = slots_ids
        self._observed = observed.BasesearchYpObserved(
            instance_provider.tier,
            replication,
            required_total_ratio=0.9,
            required_replicas_ratio=0.6,
            allow_lack_shards_ratio=0
        )

        self._porto_properties = self._default_porto_properties.copy()
        self._porto_properties.update(porto_properties or {})

    @property
    def tags(self):
        return self._instance_provider.tags

    @property
    def slots_ids(self):
        return self._slots_ids

    def update(self, reports):
        agents = frozenset(self._instance_provider.agents)
        reports = {
            agent: report.data
            for agent, report in reports.iteritems()
            if agent in agents
        }
        self._observed.update(reports)

    def gencfg(self):
        configs = {}
        for agent, instance in self._instance_provider.agents_instances.iteritems():
            slots_config = {}
            for slot_id, configurations in self._slots_targets.items():
                if slot_id in self.slots_ids:
                    slots_config[slot_id] = {
                        'instances': self._generate_slot_config(instance, configurations),
                        'default': self._slots_default_revisions.get(slot_id),
                    }
            configs[agent] = {'slots': slots_config}

        return configs

    def _generate_configuration_config(self, instance, configuration):
        return {
            'revision': configuration.revision,
            'conf_hash': hash(configuration),
            'container': self._porto_properties,
            'resources': [
                resolve_resource('basesearch.executable', configuration.base.executable),
                resolve_resource('basesearch.models', configuration.base.models),
                self._cfg_resource('basesearch.cfg', instance, configuration.base),
            ],
        }

    def _cfg_resource(self, name, instance, conf_bs):
        agent = instance.get_agent()
        resource = resolve_resource(name, conf_bs.config).copy()
        if conf_bs.config_path:
            resource['extract_file'] = conf_bs.config_path.format(
                short_host=agent.short_host, fqdn=instance.hostname, port=instance.port,
                tier=self._instance_provider.tier.name,
                podset_id=(instance.podset if hasattr(instance, 'podset') else None),
                shard_number=0,  # TODO: fix!
            )
        return resource

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._instance_provider.tier)


class RtyserverCtrl(BaseCtrl):
    def __init__(self, *args, **kwargs):
        super(RtyserverCtrl, self).__init__(*args, **kwargs)
        self._tags = kwargs.get('tags') or {'a_itype_rtyserver', 'a_tier_{}'.format(self._tier.name)}


class MmetaCompCtrl(ComponentCtrl):
    tags = {
        'a_itype_mmeta',
    }
    is_head_slot = True

    def __init__(self, slots_agents, use_dynamic_ports=False, ratio=0.75):
        super(MmetaCompCtrl, self).__init__()
        self._slots_agents = slots_agents
        self._agents = {agent for agents in slots_agents.values() for agent in agents}
        self._observed = observed.MmetaObserved(slots_agents, ratio)
        self._use_dynamic_ports = use_dynamic_ports

    @property
    def slots_ids(self):
        return self._slots_agents.keys()

    def update(self, reports):
        reports = {agent: report.data for agent, report in reports.items() if agent in self._agents}
        self._observed.update(reports)

    def _add_extra_to_config(self, instance, configuration):
        if self._use_dynamic_ports:
            return {'port': porter.conf_to_port(configuration)}
        return {}

    def gencfg(self):
        result = {}

        for slot_id, configurations in self._slots_targets.items():
            if slot_id in self._slots_agents:
                for agent in self._slots_agents[slot_id]:
                    cfg = {
                        slot_id: {
                            'instances': self._generate_slot_config(agent, configurations),
                            'default': self._slots_default_revisions.get(slot_id),
                        }
                    }
                    result[agent] = {'slots': cfg}

        return result

    def _generate_configuration_config(self, agent, configuration):
        return {
            'revision': configuration.revision,
            'conf_hash': hash(configuration),
            'resources': [
                resolve_resource('mmeta.executable', configuration.mmeta.executable),
                resolve_resource('mmeta.models', configuration.mmeta.models),
                self._cfg_resource(agent, configuration.mmeta),
            ],
        }

    @staticmethod
    def _cfg_resource(agent, conf_mmeta):
        resource = resolve_resource('httpsearch.cfg', conf_mmeta.config).copy()
        if conf_mmeta.config_path:
            resource['extract_file'] = conf_mmeta.config_path.format(short_host=agent.short_host, port=agent.port)
        return resource

    def instances_proto(self, slot_id, configuration):
        instances = []
        for agent in self._slots_agents[slot_id]:
            port = porter.conf_to_port(configuration) if self._use_dynamic_ports else agent.port
            instances.append(slot_state_pb2.InstanceInfo(hostname=agent.host, port=port))
        return instances


class SlotsCtrl(sdk.Controller):
    path = 'slots'

    def __init__(self, component_ctrls):
        super(SlotsCtrl, self).__init__()
        self._slots_targets = {}
        self._component_ctrls = component_ctrls

        self.register(*component_ctrls)
        self.add_handler('/slot_state', self._slot_state_proto_handler)

    @property
    def slots_ids(self):
        slots_ids = set()
        for component in self._component_ctrls:
            slots_ids.update(component.slots_ids)
        return list(slots_ids)

    def execute(self):
        slots_default_revisions = {
            slot_id: max(self.ready_revisions(slot_id) or [None])
            for slot_id in self._slots_targets
            if slot_id in self.slots_ids
        }
        for component in self._component_ctrls:
            component.set_target(self._slots_targets.copy(), slots_default_revisions)

    def set_slot_target(self, slot_id, configurations):
        self._slots_targets[slot_id] = configurations

    def _ready_hashes(self, slot_id):
        hashes = [set(component.ready_hashes(slot_id)) for component in self._component_ctrls
                  if slot_id in component.slots_ids]
        return set.intersection(*hashes)

    def ready_revisions(self, slot_id):
        ready_hashes = self._ready_hashes(slot_id)
        return {
            target.revision
            for target in self._slots_targets[slot_id]
            if hash(target) in ready_hashes
        }

    def json_view(self):
        result = {}
        for slot_id in self.slots_ids:
            result[slot_id] = []
            for conf in self._slots_targets.get(slot_id):
                result[slot_id].append({
                    'comps': [component.json(slot_id, hash(conf)) for component in self._component_ctrls
                              if slot_id in component.slots_ids],
                    'hash': hash(conf),
                    'conf': conf.dump_json_flat(),
                })
        return result

    def html_view(self):
        slots_ = []
        for slot_id, configurations in self._slots_targets.items():
            subs = []
            for conf in configurations:
                subs.append(
                    blocks.wrap(
                        blocks.Header('r{}'.format(conf.revision), 3),
                        blocks.Header(hash(conf), 5),
                        *[component.html(slot_id, hash(conf)) for component in self._component_ctrls
                          if slot_id in component.slots_ids]
                    )
                )
            slots_.append(blocks.wrap(blocks.Header(slot_id, 2), *subs))
        return blocks.Block(slots_)

    def slot_state_proto(self, slot_id, revision):
        if slot_id not in self._slots_targets:
            return slot_state_pb2.Response(error=slot_state_pb2.Response.UNKNOWN_SLOT)
        configuration = self._find_conf(slot_id, revision)
        if configuration is None:
            return slot_state_pb2.Response(error=slot_state_pb2.Response.UNKNOWN_REVISION)

        status = slot_state_pb2.Response.WAIT_FOR_QUORUM
        if hash(configuration) in self._ready_hashes(slot_id):
            status = slot_state_pb2.Response.READY
        return slot_state_pb2.Response(
            instances=self._mmeta_comp(slot_id).instances_proto(slot_id, configuration),
            status=status,
            components=[
                component.component_state_proto(slot_id, configuration)
                for component in self._component_ctrls
                if slot_id in component.slots_ids
            ],
        )

    @request.proto_request(request_message_type=slot_state_pb2.Request)
    def _slot_state_proto_handler(self, request_message):
        slot_id = request_message.slot_id or request.current_request().args['slot_id']
        revision = request_message.revision or request.current_request().args['revision']

        return self.slot_state_proto(slot_id, int(revision))

    def _mmeta_comp(self, slot_id):
        for component in self._component_ctrls:
            if component.is_head_slot and slot_id in component.slots_ids:
                return component
        raise RuntimeError()

    def _find_conf(self, slot_id, revision):
        for conf in self._slots_targets[slot_id]:
            if conf.revision == revision:
                return conf
        return None


def resolve_resource(resource_name, resource_id):
    if resource_id.startswith('rbtorrent:'):
        try:
            logging.debug('Try to get filenames for %s', resource_id)
            resource = {
                'url': resource_id,
                'name': resource_name,
                'file_name': skynet.get_top_filename(resource_id)
            }
            return resource
        except skynet.ResourceNotFound, skynet.Timeout:
            raise ResourceNotFound('could not get list for {}'.format(resource_id))

    if resource_id.startswith('sbr:'):
        resource_id = resource_id.split('sbr:')[1]
        resource_description = _sandbox_resource_description(resource_id)

        if resource_description is None or \
                not all((
                    resource_description['skynet_id'],
                    resource_name,
                    resource_description['file_name']
                )):
            raise ResourceNotFound('resource {} not found in sandbox'.format(resource_id))
        resource = {
            'url': resource_description['skynet_id'],
            'name': resource_name,
            'file_name': resource_description['file_name'].split('/')[-1],
        }
        return resource
    raise RuntimeError('Cannot understand resource id [{}]'.format(resource_id))


@memoize.memoized
def _sandbox_resource_description(resource_id):
    client = sandbox_utils.get_sandbox_client()
    try:
        result = client.resource[resource_id].read()
        if result and result.get('state') in ('READY',):
            return result
        return memoize.SaveWithTtl(value=None, ttl=15 * 60)
    except client.HTTPError as exc:
        if 'NOT FOUND' in exc.message:
            return memoize.SaveWithTtl(value=None, ttl=15 * 60)
        else:
            raise


class ResourceNotFound(RuntimeError):
    pass
