import datetime
import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils

import tables
import nanny_utils


class NoCommonConfig(sdk.notify.ValueNotification):
    name = 'no-common-config'
    message_template = 'Common config for {family} has been updated {value} seconds ago'
    ranges = (
        sdk.notify.Range(sdk.notify.NotifyLevels.IDLE, None, 3 * 3600),
        sdk.notify.Range(sdk.notify.NotifyLevels.INFO, 3 * 3600, 6 * 3600),
        sdk.notify.Range(sdk.notify.NotifyLevels.WARNING, 6 * 3600, None),
    )


class ServiceFamily(object):
    def __init__(self, name, resource_types, service_list):
        self._name = name
        self._resource_types = resource_types
        self._service_list = service_list
        self._last_success_time = None

    @property
    def name(self):
        return self._name

    @property
    def last_success_time(self):
        if self._last_success_time:
            return self._last_success_time
        return datetime.datetime.min

    def _get_common_resources(self, resource_getter, *args):
        services_resources = {
            service: resource_getter(service, *args)
            for service in self._service_list
        }

        resource_set = services_resources.values().pop()

        if all(resource_set == r_set for r_set in services_resources.values()):
            self._last_success_time = datetime.datetime.now()
            return _add_rbtorrents(resource_set)

        return None

    def get_common_resource_set(self):
        resource_set = self._get_common_resources(nanny_utils.get_service_resources, self._resource_types)

        if not resource_set:
            logging.warning('Common resource set not found for %s services %s', self.name, self._service_list)

        return resource_set

    def get_common_shardmap(self):
        shardmaps = [
            nanny_utils.get_service_shardmap(service)
            for service in self._service_list
        ]

        for resource in shardmaps:
            if resource != shardmaps[0]:
                logging.warning('Common shardmap not found for services %s', self._service_list)
                return None

        return shardmaps[0]


class NannyService(object):
    def __init__(self, name, resources):
        self.name = name
        self.resources = resources

    def read(self):
        return _add_rbtorrents(nanny_utils.get_service_resources(self.name, self.resources))


class WebDbChecker(object):
    def __init__(self, service_famity):
        self._service_familty = service_famity

    def get_common_timestamp(self):
        resources = self._service_familty.get_common_resource_set()

        if resources:
            assert len(resources) == 1

            vars_conf_resource = resources.values().pop()
            return self._get_yt_state(vars_conf_resource['task_id'])

        logging.info('Could not find common db timestamp for %s', self._service_familty.name)
        return None

    def _get_yt_state(self, task_id):
        task_details = sandbox_utils.get_task(task_id)
        current_db_timestamp = task_details.get('input_parameters', {}).get('db_timestamp')

        if current_db_timestamp:
            return int(current_db_timestamp)

        logging.warning('Could not get task[id=%s] details', task_id)
        return None


class Controller(sdk.Controller):
    path = 'oldprod2'

    def __init__(self, yt_table, service_families, db_checker=None):
        super(Controller, self).__init__()
        self._yt_table = yt_table
        self._service_families = service_families
        self._db_checker = db_checker

        self.common_state = None
        self.common_config = None

    @property
    def last_success_time(self):
        return min([family.last_success_time for family in self._service_families])

    def update(self, reports):
        last_status = self._yt_table.last_status()

        new_status = tables.ProductionStatus(
            self._db_checker.get_common_timestamp() if self._db_checker else None,
            dict.fromkeys(family.name for family in self._service_families)
        )

        for family in self._service_families:
            current_state = family.get_common_resource_set()

            new_status.config[family.name] = current_state if current_state else last_status.config.get(family.name)

        if last_status != new_status:
            logging.info('Status changed:\n%s\n%s', last_status, new_status)
            self._yt_table.write(new_status)

            self.common_state = new_status.db_timestamp
            self.common_config = new_status.config
        else:
            self.common_state = last_status.db_timestamp
            self.common_config = last_status.config

    def json_view(self):
        last_success_time = None

        if self.last_success_time > datetime.datetime.min:
            last_success_time = self.last_success_time.strftime('%Y-%m-%d %H:%M:%S')

        return {
            'meta': {'last_success_time': last_success_time},
            'common': {'yt_state': self.common_state, 'config': self.common_config}
        }

    def notifications(self):
        notes = []

        for family in self._service_families:
            timediff = datetime.datetime.now() - family.last_success_time
            labels = {'family': family.name}
            notes.append(
                NoCommonConfig(value=timediff.seconds, labels=labels, **labels)
            )

        return notes


def get_yt_status_table(yt_client, path, readonly=True):
    return tables.ProductionStatusTable(
        yt_client,
        path=path,
        readonly=readonly
    )


def _add_rbtorrents(resource_set):
    updated_resources = dict()

    for res_type, resource in resource_set.iteritems():
        try:
            resource_descr = sandbox_utils.get_resource_description(resource['resource_id'])
            updated_resources[res_type] = resource
            updated_resources[res_type]['skynet_id'] = resource_descr['skynet_id']
        except Exception:
            _log.exception('Could not get rbtorrent for resource %s, %s', res_type, resource)
            raise

    return updated_resources


_log = logging.getLogger(__name__)
