import logging
import collections

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.deployer.controller as deploy_controller1
import infra.callisto.controllers.deployer2.controller as deploy_controller2
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.controllers.slots.controllers as controllers
import infra.callisto.controllers.slots.state as slot_state
import infra.callisto.controllers.slots.slot as slot


def make_controller(
        yt_observer,
        basesearch_group,
        deployer_group,
        tag,
        builder_groups,
        use_new_deployer=False,
        **builder_args
):
    deployers_groups = [(deployer_group, tag)]
    if use_new_deployer:
        deploy_ctrl = deploy_controller2.make_controller(deployers_groups)
    else:
        deploy_ctrl = deploy_controller1.make_deployer_controller(deployers_groups)
    slot_ctrl = controllers.make_slot_controller(
        slot.Slot(yt_observer.name, yt_observer.tier, basesearch_group, tag),
        deploy_ctrl,
        gencfg_api.agent_shard_number_mapping(basesearch_group, tag),
        namespace_prefix=yt_observer.namespace_prefix if use_new_deployer else None,
    )
    builder = innerbuilder.make_controller(
        yt_observer,
        builder_groups,
        **builder_args
    )

    return PlainContourController(yt_observer, deploy_ctrl, slot_ctrl, builder)


class PlainContourController(sdk.Controller):
    @property
    def path(self):
        return self.yt_observer.name

    def __init__(self, yt_observer, deploy_ctrl, slot_ctrl, builder):
        super(PlainContourController, self).__init__()
        self.yt_observer = yt_observer
        self.deploy_ctrl = deploy_ctrl
        self.slot_ctrl = slot_ctrl
        self.builder = builder

        self.register(
            self.deploy_ctrl,
            self.slot_ctrl,
            self.builder,
        )

    def execute(self):
        deployer_observed, searcher_observed = self.slot_ctrl.get_observed_state()
        deploy_target, search_target = calc_slot_target_state(
            deployer_observed,
            searcher_observed,
            [slot_state.SlotState(ts) for ts in self.yt_observer.get_last_generations(2)],
        )
        self.slot_ctrl.set_target_state(deploy_target, search_target)

    def deploy_progress(self, tier_name=None):
        return self.deploy_ctrl.deploy_progress(tier_name)

    def build_progress(self, tier_name):
        return self.builder.build_progress(tier_name)

    def searchers_state(self, slot_name=None):
        assert slot_name in (self.slot_ctrl.slot.name, None)
        return {self.slot_ctrl.slot.name: self.slot_ctrl.searchers_state()}

    def json_view(self):
        view = collections.OrderedDict()
        view['contour'] = self.yt_observer.name
        view['tier'] = self.yt_observer.tier.name
        view['slot'] = self.slot_ctrl.json_view()
        view['build'] = self.builder.json_view()
        return view


def calc_slot_target_state(
    deployer_observed,
    searcher_observed,
    last_generations,
):
    if not last_generations:
        return {searcher_observed} if searcher_observed else set(), searcher_observed

    deploy_target = set(last_generations)
    search_target = last_generations[0]
    if search_target in (deployer_observed & deploy_target):
        return deploy_target, search_target

    if searcher_observed and searcher_observed not in deploy_target:
        deploy_target = {last_generations[0], searcher_observed}

    return deploy_target, searcher_observed


_log = logging.getLogger('plain_contour')
