"""
    Standard blocks for users' custom html viewers
"""


class Block(object):
    def __init__(self, subs=()):
        self._subs = subs

    def json(self):
        return {
            'type': self.__class__.__name__,
            'subs': [sub.json() for sub in self._subs],
        }


def wrap(*subs):
    return Block(subs)


class Header(Block):
    def __init__(self, text, level=3):
        super(Header, self).__init__()
        self._text = text
        self._level = level

    def json(self):
        return dict(
            super(Header, self).json(),
            text=self._text,
            level=self._level,
        )


class Href(Block):
    def __init__(self, text, url):
        super(Href, self).__init__()
        self._text = text
        self._url = url

    def json(self):
        return dict(
            super(Href, self).json(),
            text=self._text,
            url=self._url,
        )


class HrefList(Block):
    def __init__(self, subs):
        super(HrefList, self).__init__(subs)


class Progress(Block):
    def __init__(self, done, total, timestamp=None, style=None):
        super(Progress, self).__init__()
        self._done = done
        self._total = total
        self._timestamp = timestamp
        self._style = style
        assert style in ('success', 'info', None)

    def json(self):
        return dict(
            super(Progress, self).json(),
            done=self._done,
            total=self._total,
            timestamp=self._timestamp,
            style=self._style,
        )


class DoubleProgress(Block):
    def __init__(self, done, total, done_overall, total_overall, timestamp=None):
        super(DoubleProgress, self).__init__([Progress(done, total), Progress(done_overall, total_overall)])
        self._timestamp = timestamp

    def json(self):
        return dict(
            super(DoubleProgress, self).json(),
            timestamp=self._timestamp,
        )


class BuilderView(Block):
    def __init__(self, bars, name=None, href_list=None):
        super(BuilderView, self).__init__(bars)
        self._name = name
        self._href_list = href_list or HrefList([])

    def json(self):
        return dict(
            super(BuilderView, self).json(),
            name=self._name,
            href_list=self._href_list.json(),
        )


class SearchView(BuilderView):
    pass


class DeployView(BuilderView):
    pass


class SlotView(Block):
    def __init__(self, name=None, deploy_view=None, search_view=None):
        super(SlotView, self).__init__()
        self._name = name
        self._deploy = deploy_view
        self._search = search_view

    def json(self):
        return dict(
            super(SlotView, self).json(),
            name=self._name,
            deploy=self._deploy.json(),
            search=self._search.json(),
        )


class Intl2Target(Block):
    def __init__(self, slots_targets):
        super(Intl2Target, self).__init__()
        self._slots_targets = slots_targets

    def json(self):
        return dict(
            super(Intl2Target, self).json(),
            slots_targets=self._slots_targets,
        )


class Intl2View(Block):
    def __init__(self, target_state, progress):
        super(Intl2View, self).__init__()
        self._target = target_state
        self._progress = progress

    def json(self):
        return dict(
            super(Intl2View, self).json(),
            target=self._target.json(),
            progress=self._progress.json(),
        )


class Banner(Block):
    def __init__(self, text, style='info'):
        super(Banner, self).__init__()
        self._text = text
        self._style = style
        assert style in ('success', 'info', 'warning', 'danger')

    def json(self):
        return dict(
            super(Banner, self).json(),
            text=self._text,
            style=self._style,
        )


class YtTimestamp(Block):
    def __init__(self, timestamp):
        super(YtTimestamp, self).__init__()
        self._timestamp = timestamp

    def json(self):
        return dict(
            super(YtTimestamp, self).json(),
            timestamp=self._timestamp,
        )


class Shard(Block):
    def __init__(self, shard):
        super(Shard, self).__init__()
        self._shard = shard

    def json(self):
        return dict(
            super(Shard, self).json(),
            shard=self._shard,
        )


class TableTierView(BuilderView):
    pass


class InnerTableRow(Block):
    def __init__(self, cols, col_width=None):
        super(InnerTableRow, self).__init__()
        self._cols = cols
        default_col_width = 12 / len(cols)
        self._col_width = col_width or [default_col_width for _ in xrange(len(cols))]

    def json(self):
        return dict(
            super(InnerTableRow, self).json(),
            cols=[col.json() for col in self._cols],
            col_width=self._col_width,

        )


class TableView(Block):
    def __init__(self, blocks, header_num=None, col_width=None):
        super(TableView, self).__init__()
        self._blocks = blocks
        self._header_num = header_num
        default_col_width = 12 / len(blocks)
        self._col_width = col_width or [default_col_width for _ in xrange(len(blocks))]

    def json(self):
        return dict(
            super(TableView, self).json(),
            header_num=self._header_num,
            blocks=[[col.json() for col in row] for row in self._blocks],
            col_width=self._col_width,
        )
