import infra.callisto.controllers.core.links as links
import blocks


class Controller(object):
    """
        define `tags` to pass reports to update()
        define `reports_alive_threshold` to filter old reports
        define `id` to use load/store context
            (should be unique underscore-separated string)
        define `path` or `id` to access child ctrl via its path in controllers tree
    """
    reports_alive_threshold = None

    @property
    def id(self):
        return None

    @property
    def path(self):
        return self._path or self.id

    @path.setter
    def path(self, value):
        self._path = value

    @property
    def tags(self):
        return None

    def __init__(self):
        self._handlers = {}
        self._path = None
        links.register(self)

    def register(self, *controllers):
        for ctrl in controllers:
            links.add_link(self, ctrl)

    def add_handler(self, path, func):
        self._handlers[path] = func

    # loop methods:

    def set_context(self, context):
        pass

    def update(self, reports):
        pass

    def execute(self):
        pass

    def gencfg(self):
        pass

    # actually it is a hack, needed due to architecture problems with update/execute scheme
    # useful to write to some table after execute() stage of all the controllers in the tree
    def save_status(self):
        pass

    def get_context(self):
        pass

    # viewers:

    def json_view(self):
        return None

    def html_view(self):
        views = filter(None, (child.html_view() for child in self.children))
        return blocks.Block(views)

    @classmethod
    def notifications(cls):
        return []

    # other:

    @property
    def handlers(self):
        return self._handlers.copy()

    @property
    def children(self):
        return links.children(self)

    def __str__(self):
        return self.__class__.__name__

    def __repr__(self):
        return str(self)

    def _repr_pretty_(self, printer, cycle):
        """
        pretty printer for ipython
        """
        if cycle:
            printer.text('{} (...)'.format(self.__class__.__name__))
        else:
            printer.text('{} (tags={})'.format(self.__class__.__name__, self.tags))
