import infra.callisto.controllers.utils.const_obj as const_obj
import blocks


class _NotifyLevel(object):
    def __init__(self, str_, num):
        self._str = str_
        self._num = num

    def __lt__(self, other):
        return self._num < getattr(other, '_num')

    def __le__(self, other):
        return self._num <= getattr(other, '_num')

    def __gt__(self, other):
        return self._num > getattr(other, '_num')

    def __ge__(self, other):
        return self._num >= getattr(other, '_num')

    def __eq__(self, other):
        return self._str == getattr(other, '_str')

    def __str__(self):
        return self._str


class NotifyLevels(object):
    IDLE = _NotifyLevel('IDLE', 0)
    INFO = _NotifyLevel('INFO', 10)
    WARNING = _NotifyLevel('WARNING', 20)
    ERROR = _NotifyLevel('ERROR', 30)

    ALL = [IDLE, INFO, WARNING, ERROR]

    @classmethod
    def banner_style(cls, level):
        if level == cls.WARNING:
            return 'warning'
        elif level == cls.ERROR:
            return 'danger'
        return 'info'


class Range(const_obj.ConstObj):
    def __init__(self, level, left_border, right_border):
        self.level = level
        self.left = left_border
        self.right = right_border

        self.on_initialize()

    def __contains__(self, item):
        if self.left is not None and item < self.left:
            return False
        if self.right is not None and self.right <= item:
            return False
        return True


class Notification(object):
    @property
    def level(self):
        raise NotImplementedError()

    @property
    def message(self):
        raise NotImplementedError()

    def as_banner(self):
        return blocks.Banner(self.message, NotifyLevels.banner_style(self.level))

    def json(self):
        return {
            'message': self.message,
            'level': str(self.level),
        }


class TextNotification(Notification):
    def __init__(self, message, level):
        self._message = message
        self._level = level

    @property
    def message(self):
        return self._message

    @property
    def level(self):
        return self._level


class ValueNotification(Notification):
    name = None
    message_template = None
    ranges = [Range(NotifyLevels.IDLE, None, None)]

    def __init__(self, value, labels=None, **template_args):
        self._value = value
        self._template_args = template_args
        self._labels = labels or {}
        assert self.name

    @property
    def value(self):
        return self._value

    @property
    def level(self):
        for range_ in self.ranges:
            if self.value in range_:
                return range_.level
        return NotifyLevels.IDLE

    @property
    def message(self):
        if self.message_template is not None:
            return self.message_template.format(value=self.value, **self._template_args)
        labels = ', '.join(['{}: {}'.format(k, v) for k, v in sorted(self._labels.items())])
        if labels:
            labels = ' [' + labels + ']'
        return '{}{}: {}'.format(self.name, labels, self._value)

    def solomon(self):
        return {
            'labels': dict(self._labels, sensor=self.name),
            'value': self.value
        }


class NotificationsAggregator(object):
    def __init__(self):
        self._buffer = []
        self._notifications = []

    def add_notification(self, notification):
        self._buffer.append(notification)

    def add_notifications(self, notifications):
        self._buffer.extend(notifications)

    def get_notifications(self):
        return self._notifications

    def push_buffer(self):
        self._notifications = self._buffer
        self._buffer = []
