import collections

import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.funcs as funcs


_AcceptanceDump = collections.namedtuple('AcceptanceDump', ['accepted', 'skipped'])


def _make_dump(skipped, accepted):
    skipped = frozenset(skipped)
    accepted = frozenset(accepted)
    if skipped & accepted:
        raise ValueError('{} timestamps are accepted and skipped at once'.format(skipped & accepted))
    return _AcceptanceDump(accepted=accepted, skipped=skipped)


class AcceptanceStaticSource(object):
    def __init__(self, skipped, accepted):
        self._dump = _make_dump(skipped=skipped, accepted=accepted)

    def load(self):
        return self._dump


class AcceptanceTableSource(yt_utils.SortedYtTable):
    def _load_rows(self):
        request = '* from [{}] order by (State, Timestamp) desc limit 10000'.format(self.path)
        return list(self._select_rows(request))

    def load(self):
        last_row_for_timestamp = {}
        for row in sorted(self._load_rows(), key=lambda x: x['Timestamp']):
            timestamp = funcs.yt_state_to_timestamp(row['State'])
            last_row_for_timestamp[timestamp] = row

        skipped, accepted = set(), set()
        for timestamp, row in last_row_for_timestamp.items():
            if row['IsSkipped']:
                skipped.add(timestamp)
            if row['IsAccepted']:
                accepted.add(timestamp)

        return _make_dump(skipped=skipped, accepted=accepted)
