

class DeployRuler(object):
    def __init__(self, build_source, acceptance_source, max_states_to_keep=2):
        self._build_source = build_source
        self._acceptance_source = acceptance_source
        self._max_states_to_keep = max_states_to_keep

    def to_deploy_on_prod(self, prod_timestamp):
        all_timestamps = {target.generation for target in self._build_source.get_targets()}
        skipped_timestamps = self._acceptance_source.load().skipped
        return to_deploy_on_prod(
            all_timestamps=all_timestamps,
            skipped_timestamps=skipped_timestamps,
            current_prod_timestamp=prod_timestamp,
            max_timestamps_to_keep=self._max_states_to_keep,
        )

    def to_deploy_on_acceptance(self, prod_timestamp):
        all_timestamps = {target.generation for target in self._build_source.get_targets()}
        skipped_timestamps = self._acceptance_source.load().skipped
        return to_deploy_on_acceptance(
            all_timestamps=all_timestamps,
            skipped_timestamps=skipped_timestamps,
            current_prod_timestamp=prod_timestamp,
            max_timestamps_to_keep=self._max_states_to_keep,
        )

    @classmethod
    def search_target_on_acceptance(cls, to_deploy, prod_common_timestamp):
        newer_than_prod = [x for x in to_deploy if x > prod_common_timestamp]
        if newer_than_prod:
            return min(newer_than_prod)
        else:
            return prod_common_timestamp


def check_input(all_timestamps, skipped_timestamps, prod_timestamp):
    assert prod_timestamp in all_timestamps, 'prod timestamp {} not in build target table'.format(prod_timestamp)
    assert prod_timestamp not in skipped_timestamps, 'prod timestamp {} is skipped'.format(prod_timestamp)


def to_deploy_on_prod(all_timestamps, skipped_timestamps, current_prod_timestamp, max_timestamps_to_keep):
    check_input(all_timestamps, skipped_timestamps, current_prod_timestamp)

    not_skipped_timestamps = sorted(set(all_timestamps) - set(skipped_timestamps), reverse=True)
    prod_index = not_skipped_timestamps.index(current_prod_timestamp)
    if prod_index < max_timestamps_to_keep:
        result = not_skipped_timestamps[:max_timestamps_to_keep]
    else:
        result = not_skipped_timestamps[prod_index - max_timestamps_to_keep + 1: prod_index + 1]

    assert current_prod_timestamp in result
    return set(result)


def to_deploy_on_acceptance(all_timestamps, skipped_timestamps, current_prod_timestamp, max_timestamps_to_keep):
    check_input(all_timestamps, skipped_timestamps, current_prod_timestamp)

    not_skipped_timestamps = sorted(set(all_timestamps) - set(skipped_timestamps))
    prod_index = not_skipped_timestamps.index(current_prod_timestamp)
    candidate_timestamps = not_skipped_timestamps[prod_index + 1 :]
    return set(candidate_timestamps[:max_timestamps_to_keep] or [current_prod_timestamp])
