import logging
import datetime
import collections

import aniso8601

import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils
import infra.callisto.controllers.utils.funcs as funcs

from .. import notify


SwitchTask = collections.namedtuple('SwitchTask', [
    'id', 'index_generation', 'url', 'is_released', 'finished',
])


def find_switch_task(task_type, vertical_name, index_generation):
    client = sandbox_utils.get_sandbox_client()
    tags = [
        _vertical_tag(vertical_name),
    ]
    tasks_found = client.task.read(
        type=task_type,
        limit=10,
        tags=tags,
        all_tags=True,
        hints=[index_generation],
        all_hints=True
    )
    if tasks_found.get('total') == 0:
        return None

    for task in tasks_found['items']:
        if task['input_parameters']['db_timestamp'] == index_generation:
            finish = aniso8601.parse_datetime(task['execution']['finished']) if 'finished' in task['execution'] else None
            return SwitchTask(
                id=task['id'],
                index_generation=index_generation,
                url=_task_gui_url(task['id']),
                is_released=sandbox_utils.is_task_released(task),
                finished=finish,
            )

    raise RuntimeError('no task with tags {} has correct db_timestamp parameter'.format(tags))


def make_switch_task(task_type, vertical_name, index_generation, auto_release=False):
    client = sandbox_utils.get_sandbox_client()
    task_id = client.task({'type': task_type})['id']

    description = '{} db index {} (aka {})'.format(
        vertical_name,
        index_generation,
        funcs.timestamp_to_yt_state(index_generation),
    )
    custom_fields = [{'name': 'db_timestamp', 'value': index_generation}]

    if auto_release:
        custom_fields.append({'name': 'auto_release', 'value': auto_release})

    client.task[task_id] = {
        'owner': 'SEPE-PRIEMKA',
        'description': description,
        'priority': {
            'subclass': 'HIGH',
            'class': 'SERVICE'
        },
        'custom_fields': custom_fields,
        'tags': [
            _vertical_tag(vertical_name),
        ],
        'hints': [index_generation]
    }
    client.batch.tasks.start.update(task_id)
    _log.info('created %s task. id %s, db timestamp %s', task_type, task_id, index_generation)
    return task_id


def _vertical_tag(vertical_name):
    return 'vertical_{}'.format(vertical_name)


class Switcher(object):
    def __init__(
        self,
        search_ruler,
        task_type,
        vertical_name,
        should_be_deployed_time=datetime.time(hour=21),
        should_be_accepted_time=datetime.time(hour=23),
        deadline_time=datetime.time(hour=1),
        alerting_interval=datetime.timedelta(seconds=3600),
        enable_autorelease=False,
        readonly=True,
    ):
        self._search_ruler = search_ruler
        self._task_type = task_type
        self._vertical_name = vertical_name
        self._should_be_deployed_time = should_be_deployed_time
        self._should_be_accepted_time = should_be_accepted_time
        self._deadline_time = deadline_time
        self._alerting_interval = alerting_interval
        self._enable_autorelease = enable_autorelease
        self.readonly = readonly

    def _need_check_deploy(self):
        return (
            datetime.datetime.now().time() >= self._should_be_deployed_time
            or datetime.datetime.now().time() <= self._deadline_time
        )

    def _need_check_acceptance(self):
        return (
            datetime.datetime.now().time() >= self._should_be_accepted_time
            or datetime.datetime.now().time() <= self._deadline_time
        )

    def _may_create_task(self):
        return self._need_check_acceptance()

    def _is_alerting_time(self):
        now = datetime.datetime.now()
        return (
            (now + self._alerting_interval).time() > self._deadline_time
            and now.time() <= self._deadline_time
        )

    def check_and_switch(self, current_timestamp, unistat_signals={}):
        unistat_signals.update({
            'db_not_deployed_axxx': 0,
            'db_not_accepted_axxx': 0,
            'task_not_released_axxx': 0,
            'critical_time_axxx': 0,
        })

        notifications = []
        next_timestamp = self._search_ruler.next_timestamp(current_timestamp)
        if not next_timestamp:
            return notifications

        is_deployed = self._search_ruler.is_deployed(next_timestamp)
        is_accepted = self._search_ruler.is_accepted(next_timestamp)
        existing_task = find_switch_task(self._task_type, self._vertical_name, next_timestamp)

        if not is_deployed:
            unistat_signals['db_not_deployed_axxx'] = 1
        if not is_accepted:
            unistat_signals['db_not_accepted_axxx'] = 1
        if not existing_task or not existing_task.is_released:
            unistat_signals['task_not_released_axxx'] = 1
        if self._is_alerting_time():
            unistat_signals['critical_time_axxx'] = 1

        # warn if task is created, but base is not deployed/accepted for some reason
        if not is_deployed and (self._need_check_deploy()):
            notifications.append(notify.TextNotification(
                '{} is still not deployed'.format(next_timestamp),
                notify.NotifyLevels.WARNING,
            ))
        if not is_accepted and (self._need_check_acceptance()):
            notifications.append(notify.TextNotification(
                '{} is still not accepted'.format(next_timestamp),
                notify.NotifyLevels.WARNING,
            ))

        if existing_task:
            _log.debug('found existing task %s', existing_task)
            if not existing_task.is_released:
                notifications.append(notify.TextNotification(
                    'created switch task {} ({})'.format(existing_task.id, existing_task.url),
                    notify.NotifyLevels.WARNING if self._may_create_task() else notify.NotifyLevels.INFO
                ))
        elif self._may_create_task() and is_deployed and is_accepted:
            if not self.readonly:
                _log.info('no task found, create new one')
                make_switch_task(self._task_type, self._vertical_name, next_timestamp,
                                 self._enable_autorelease)
            else:
                _log.info('Skip %s task create for %s timestamp %s due to RO mode',
                          self._task_type, self._vertical_name, next_timestamp)

        return notifications


def _task_gui_url(task_id):
    return 'https://sandbox.yandex-team.ru/task/{}/view'.format(task_id)


_log = logging.getLogger(__name__)
