"""
    Register root-controllers (entry points)
"""

import inspect
import collections

import infra.callisto.context.mongo as context
import infra.callisto.configs.mongo as configs


ROOT_CTRLS = {}


_Registry = collections.namedtuple('_Registry', [
    'root_ctrl_path',
    'make_ctrl',
    'reports_backends',
    'context_storage',
    'configs_storage',
    'sleep_time',
    'config_type',
])


class ReportsBackends(object):
    class Default(object):
        sas = 'http://sas-reports-cajuper.n.yandex-team.ru'
        testing = 'http://man-reports-testing-cajuper.n.yandex-team.ru'
        all = (sas,)

    class V2(object):
        man = 'http://man-reports-v2-cajuper.n.yandex-team.ru'
        vla = 'http://vla-reports-v2-cajuper.n.yandex-team.ru'
        sas = 'http://sas-reports-v2-cajuper.n.yandex-team.ru'
        all = (sas, vla)
        # all = (man, sas, vla)


def register(
        root_ctrl_path,
        make_ctrl_function,
        reports_backends,
        sleep_time=180,
):
    assert root_ctrl_path not in ROOT_CTRLS
    assert inspect.getargspec(make_ctrl_function).args == ['readonly'], 'make_ctrl must have 1 arg `readonly`'

    old_style_name = NEW_TO_OLD_NAME_MAPPING.get(root_ctrl_path, root_ctrl_path)
    context_storage_factory = context.require(old_style_name)
    configs_storage_factory = configs.require(old_style_name)
    ROOT_CTRLS[root_ctrl_path] = _Registry(
        root_ctrl_path,
        make_ctrl_function,
        tuple(reports_backends),
        context_storage_factory, configs_storage_factory,
        sleep_time=sleep_time,
        config_type=None
    )
    return ROOT_CTRLS[root_ctrl_path]


def register2(controller_path, factory, sleep_time=180, config_type=None):
    """
    Register controller which doesn't require mongo-based configs and reports facilities.

    :param str controller_path: unique human-readable identifier, e.g. web/prod/vla
    :param (bool) -> Any factory: controller creation function with single parameter `readonly`
    :param sleep_time: interval between loop iterations
    :param config_type: type of controller's private protoconfig
    """
    assert controller_path not in ROOT_CTRLS, 'Controller [{}] already registered'.format(controller_path)
    assert inspect.getargspec(factory).args[:1] == ['readonly'], \
        'Factory function must have the first argument `readonly`'
    if config_type is not None:
        from google.protobuf.message import Message
        assert issubclass(config_type, Message), 'config_type must be a protobuf message type, if specified'

    ROOT_CTRLS[controller_path] = _Registry(controller_path, factory, None, None, None,
                                            sleep_time=sleep_time, config_type=config_type)


def init_users():
    import importlib
    importlib.import_module('infra.callisto.controllers.user')


NEW_TO_OLD_NAME_MAPPING = {
    'web/prod/main': 'jupiter',
    'web/prod/man': 'man_web',
    'web/prod/sas': 'sas_web',
    'web/prod/vla': 'vla_web',
    'web/prod/acceptance': 'pip',
    'web/callisto/prod/man': 'ManCallisto',
    'web/callisto/prod/sas': 'SasCallisto',
    'web/callisto/prod/vla': 'VlaCallisto',
    'web/betas/multi/base': 'multi_beta',
    'web/betas/multi/int': 'multi_int',
    'web/betas/multi/meta': 'multi_meta',
    'web/betas/nightly/test': 'NightlyTest',
    'web/betas/nightly/baseline': 'NightlyBaseline',

    'images/prod/build/tier1': 'ImgTier1',
    'images/prod/thumb/wide': 'ImgThumbWide',
    'images/prod/rim': 'ImgRIMTier',
    'images/betas/beta1': 'ImgBeta',
    'images/betas/beta2': 'ImgBeta2',
    'images/betas/multi/base': 'imgs_multi_beta',
    'images/betas/multi/meta': 'imgs_multi_meta',

    'video/prod/main': 'video',
    'video/prod/man': 'man_video',
    'video/prod/sas': 'sas_video',
    'video/prod/vla': 'vla_video',
    'video/prod/acceptance': 'pip_video',
    'video/betas/multi/base': 'video_multi_beta',
    'video/betas/multi/meta': 'video_multi_meta',

    'saas/betas/frozen': 'SaasFrozenBeta',
}
