# coding=utf8

import datetime
import aniso8601

import infra.callisto.libraries.yt as yt_utils

import record


Target = record.Record


class Locked(RuntimeError):
    pass


class _KeyWords(object):
    Id = 'Id'
    Locked = 'Locked'
    Time = 'Time'
    User = 'User'
    Comment = 'Comment'


def _add_special_fields_to_schema(schema):
    prefix = [
        {'name': _KeyWords.Id,      'type': 'int64', 'sort_order': 'ascending'},  # for conflicts in transactions
        {'name': _KeyWords.Locked,  'type': 'boolean', 'required': True},
    ]
    suffix = [
        {'name': _KeyWords.Time,    'type': 'string', 'required': True},
        {'name': _KeyWords.User,    'type': 'string'},
        {'name': _KeyWords.Comment, 'type': 'string'},
    ]
    return prefix + schema + suffix


def _make_target_wrapper(target_class_):
    assert issubclass(target_class_, Target)
    assert target_class_.schema is not None

    class TargetWrapper(object):
        target_class = target_class_
        schema = _add_special_fields_to_schema(target_class_.schema)

        def __init__(self, target, id, locked, time, user, comment):
            self.target = target
            self.id = id
            self.locked = locked
            self.time = time
            self.user = user
            self.comment = comment

        def dump_row(self):
            row = self.target.dump_row()
            row[_KeyWords.Id] = self.id
            row[_KeyWords.Locked] = self.locked
            row[_KeyWords.Time] = self.time.isoformat()
            row[_KeyWords.User] = self.user
            row[_KeyWords.Comment] = self.comment
            return row

        @classmethod
        def load_row(cls, row):
            id_ = row.pop(_KeyWords.Id)
            locked = row.pop(_KeyWords.Locked)
            time = aniso8601.parse_datetime(row.pop(_KeyWords.Time))
            user = row.pop(_KeyWords.User)
            comment = row.pop(_KeyWords.Comment)
            target = cls.target_class.load_row(row)
            return cls(target, id_, locked, time, user, comment)

        def __repr__(self):
            return 'Wrapper[{}]({!r})'.format(
                'id={}, locked={}'.format(self.id, self.locked),
                self.target,
            )

    return TargetWrapper


class _ControlTableMetaClass(type):
    abstract_class_created = False

    def __new__(mcs, name, bases, dct):
        class_ = super(_ControlTableMetaClass, mcs).__new__(mcs, name, bases, dct)
        if not mcs.abstract_class_created:
            mcs.abstract_class_created = True
        else:
            target_class = getattr(class_, 'target_class')
            target_wrapper_class = _make_target_wrapper(target_class)
            setattr(class_, '_target_wrapper_class', target_wrapper_class)
            setattr(class_, 'schema', target_wrapper_class.schema[:])
        return class_


class ControlTable(yt_utils.SortedYtTable):
    """override `target_class` field with subclass of Target"""
    __metaclass__ = _ControlTableMetaClass
    _target_wrapper_class = None
    target_class = None

    def _on_init_hook(self):
        self.ensure_table()
        self.ensure_mounted()

    def head(self):
        for target in self.get_last_n(1):
            return target

    def write(self, target, lock=False, override=False, comment=None):
        # user = self._yt_client.get_user_name()
        user = yt_utils.create_yt_client(self._yt_client.config['proxy']['url']).get_user_name()
        # ^ temporarily rpc yt client cannot make get_user_name request, so hack

        with self._transaction():
            head = self.head()
            if not override and head and head.locked:
                raise Locked('table is locked (row #{})'.format(head.id))
            wrapped = self._target_wrapper_class(
                target=target,
                id=(head and head.id + 1) or 0,
                locked=lock,
                time=datetime.datetime.now(),
                user=user,
                comment=comment,
            )
            self._insert_rows([wrapped.dump_row()])

    def unlock(self, comment='unlock'):
        head = self.head()
        if not head:
            raise RuntimeError('table is empty')
        if not head.locked:
            raise RuntimeError('table is not locked')
        self.write(target=head.target, override=True, comment=comment)

    def get_last_n(self, count):
        result = []
        request = '* from [{}] order by Id desc limit {}'.format(self._path, count)
        for row in self._select_rows(request):
            result.append(self._target_wrapper_class.load_row(row))
        return result
