# coding=utf8

import datetime
import aniso8601

import infra.callisto.libraries.yt as yt_utils

import record


Status = record.Record


class _KeyWords(object):
    Id = 'Id'
    Time = 'Time'


def _add_special_fields_to_schema(schema):
    prefix = [
        {'name': _KeyWords.Id, 'type': 'int64', 'sort_order': 'ascending'},
        {'name': _KeyWords.Time, 'type': 'string', 'sort_order': 'ascending'},
    ]
    return prefix + schema


def _make_wrapper(row_class_):
    assert issubclass(row_class_, Status)
    assert row_class_.schema is not None

    class StatusWrapper(object):
        row_class = row_class_
        schema = _add_special_fields_to_schema(row_class.schema)

        def __init__(self, status, id, time):
            self.status = status
            self.id = id
            self.time = time

        def dump_row(self):
            row = self.status.dump_row()
            row[_KeyWords.Id] = self.id
            row[_KeyWords.Time] = self.time.isoformat()
            return row

        @classmethod
        def load_row(cls, row):
            time = aniso8601.parse_datetime(row.pop(_KeyWords.Time))
            id_ = row.pop(_KeyWords.Id)
            status = cls.row_class.load_row(row)
            return cls(status, id_, time)

    return StatusWrapper


class _StatusTableMetaClass(type):
    abstract_class_created = False

    def __new__(mcs, name, bases, dct):
        class_ = super(_StatusTableMetaClass, mcs).__new__(mcs, name, bases, dct)
        if not mcs.abstract_class_created:
            mcs.abstract_class_created = True
        else:
            status_class = getattr(class_, 'status_class')
            wrapper_class = _make_wrapper(status_class)
            setattr(class_, 'status_wrapper_class', wrapper_class)
            setattr(class_, 'schema', wrapper_class.schema[:])
        return class_


class StatusTable(yt_utils.SortedYtTable):
    __metaclass__ = _StatusTableMetaClass
    status_class = None
    status_wrapper_class = None

    def _on_init_hook(self):
        self.ensure_table()
        self.ensure_mounted()

    def head(self):
        request = '* from [{}] order by Id desc limit 1'.format(self._path)
        for row in self._select_rows(request):
            return self.status_wrapper_class.load_row(row)

    def write(self, status):
        with self._transaction():
            head = self.head()
            wrapped = self.status_wrapper_class(
                status=status,
                id=(head and head.id + 1) or 0,
                time=datetime.datetime.now(),
            )
            self._insert_rows([wrapped.dump_row()], format='yson')
