"""
Gencfg tiers and its shards definition
Must strictly correspond to gencfg data. Otherwise everything goes wild.
"""
import math
import numbers

import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.libraries.memoize as memoize


TIERS = {}


@memoize.get_cache(records_limit=100 * 1000)
def parse_shard(shard_name):
    parsed_shards = []
    for tier in TIERS.values():
        try:
            parsed_shards.append(tier.parse_shard(shard_name))
        except (AssertionError, RuntimeError, KeyError, ValueError):
            pass
    if len(parsed_shards) == 1:
        return parsed_shards.pop()
    raise ValueError('cannot parse shardname %s: %i possible variants' % (shard_name, len(parsed_shards)))


class _Tier(object):
    class _Shard(entities.Shard):
        pass

    def __init__(
        self,
        name,
        shards_in_group,
        shard_size_gb,
        groups_count=1,
        prefix=None,
    ):
        self._name = name
        self._groups_count = groups_count
        self._shards_in_group = shards_in_group
        self._shard_size = shard_size_gb * (1024 ** 3)
        self._prefix = prefix

        assert name not in TIERS, 'Duplicate tier name {}'.format(name)
        TIERS[name] = self

    @property
    def name(self):
        return self._name

    @property
    def prefix(self):
        return self._prefix

    @property
    def shards_in_group(self):
        return self._shards_in_group

    @property
    def groups_count(self):
        return self._groups_count

    @property
    def shards_count(self):
        return self.shards_in_group * self.groups_count

    shard_count = shards_count

    @property
    def shard_size(self):
        return self._shard_size

    def make_shard(self, number, timestamp):
        if isinstance(number, basestring):
            group_number, shard_number = self._Shard.parse_number(number)
        elif isinstance(number, (tuple, list)):
            group_number, shard_number = number
        elif isinstance(number, numbers.Number):
            group_number, shard_number = 0, number
        else:
            raise ValueError('cannot make shard ' + str(type(number)))
        return self._Shard(self, group_number, shard_number, timestamp)

    def parse_shard(self, shard_fullname):
        return self._Shard.parse_fullname(self, shard_fullname)

    def list_shard_numbers(self):
        return [
            (i, j)
            for i in xrange(self.groups_count)
            for j in xrange(self.shards_in_group)
        ]

    def list_shards(self, timestamp):
        if isinstance(timestamp, basestring) and '-' in timestamp:
            timestamp = funcs.yt_state_to_timestamp(timestamp)

        return [
            self.make_shard((i, j), timestamp)
            for i, j in self.list_shard_numbers()
        ]

    def __str__(self):
        return self.name

    def __repr__(self):
        return str(self)


class _JupiterTier(_Tier):
    class _Shard(_Tier._Shard):
        shard_template = '{prefix}-{tier_name}-{group_number}-{shard_number}-{timestamp:010}'
        shard_template_without_timestamp = '{prefix}-{tier_name}-{group_number}-{shard_number}'
        number_template = '{group_number}-{shard_number}'

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if tier.name in shard_fullname:
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if (
                    params.get('tier_name') == tier.name
                    and params.get('prefix') == tier.prefix
                ):
                    return tier.make_shard(
                        (params['group_number'], params['shard_number']),
                        params['timestamp'],
                    )
            raise ValueError('cannot parse shard')


class _PrefixLessJupiterTier(_JupiterTier):
    class _Shard(_JupiterTier._Shard):
        shard_template = '{tier_name}-{group_number}-{shard_number}-{timestamp:010}'


class _JupiterMsTier(_JupiterTier):
    class _Shard(_Tier._Shard):
        shard_template = '{prefix}-jupiter-msuserdata-{shard_number:03}-{timestamp:010}'
        number_template = 'msuserdata-{shard_number:03}'

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if shard_fullname.startswith(tier.prefix):
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if params['prefix'] == tier.prefix:
                    return tier.make_shard(
                        (0, params['shard_number']),
                        params['timestamp'],
                    )
            raise ValueError('cannot parse shard')


class _ImgVideoTier(_Tier):
    class _Shard(_Tier._Shard):
        shard_template = '{prefix}-{shard_number:03}-{date}-{time}'
        shard_template_without_timestamp = '{prefix}-{shard_number:03}'
        number_template = '{shard_number:03}'

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if shard_fullname.startswith(tier.prefix):
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if params['prefix'] == tier.prefix:
                    timestamp = funcs.yt_state_to_timestamp('{}-{}'.format(params['date'], params['time']))
                    return tier.make_shard(
                        (0, params['shard_number']),
                        timestamp,
                    )
            raise ValueError('cannot parse shard')


class _ImgVideoGroupedTier(_Tier):
    class _Shard(_Tier._Shard):
        shard_template = '{prefix}-{group_number:03}-{shard_number:03}-{date}-{time}'
        number_template = '{group_number:03}-{shard_number:03}'

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if shard_fullname.startswith(tier.prefix):
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if params['prefix'] == tier.prefix:
                    timestamp = funcs.yt_state_to_timestamp('{}-{}'.format(params['date'], params['time']))
                    return tier.make_shard(
                        (params['group_number'], params['shard_number']),
                        timestamp,
                    )
            raise ValueError('cannot parse shard')


class _GeminiTier(_Tier):
    class _Shard(_Tier._Shard):
        shard_template = '{tier_name}-{shard_number}-{date}-{time}'
        number_template = '{shard_number}'

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if shard_fullname.startswith(tier.name):
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if params['tier_name'] == tier.name:
                    timestamp = funcs.yt_state_to_timestamp('{}-{}'.format(params['date'], params['time']))
                    return tier.make_shard(
                        (0, params['shard_number']),
                        timestamp,
                    )
            raise ValueError('cannot parse shard')

    @staticmethod
    def shard_number_to_range(number):
        magic_number = 1574
        number = int(number)
        if number < magic_number:
            a = number * 31
            return a, a + 30
        else:
            a = magic_number + 30 * number
            return a, a + 29


class _SaasFrozen3dayTier(_Tier):
    _magic_number = 65533

    class _Shard(_Tier._Shard):
        shard_template = '{tier_name}-{range_begins}-{range_ends}-{timestamp}'
        number_template = '{shard_number}'

        @property
        def fullname(self):
            range_begins, range_ends = self.tier.shard_number_to_range(self.shard_number)

            return self.shard_template.format(
                tier_name=self.tier.name,
                range_begins=range_begins,
                range_ends=range_ends,
                timestamp=self.timestamp,
            )

        @classmethod
        def parse_fullname(cls, tier, shard_fullname):
            if shard_fullname.startswith(tier.name):
                params = funcs.parse_by_template(shard_fullname, cls.shard_template)
                if params['tier_name'] == tier.name:
                    return tier.make_shard(
                        (
                            0,
                            tier.range_to_shard_number(int(params['range_begins']), int(params['range_ends']))
                        ),
                        params['timestamp'],
                    )
            raise ValueError('cannot parse shard')

    def shard_number_to_range(self, number):
        """
        ShardMin : ${ tostring(math.floor(65533 * shardid / SHARDS_NUMBER)) }
        ShardMax : ${ tostring(math.floor(65533 * (shardid + 1) / SHARDS_NUMBER) - 1 + math.floor((shardid + 1) / SHARDS_NUMBER) ) }
        """

        assert number < self._shards_in_group

        return [
            self._range_begins(number),
            self._range_ends(number)
        ]

    def _range_begins(self, shard_number):
        return (self._magic_number * shard_number / self._shards_in_group)

    def _range_ends(self, shard_number):
        return (self._magic_number * (shard_number + 1) / self._shards_in_group) + ((shard_number + 1) / self._shards_in_group) - 1

    def range_to_shard_number(self, range_begins, range_ends):
        shard_number = int(math.ceil(float(range_begins * self._shards_in_group) / self._magic_number))

        assert shard_number < self._shards_in_group
        range_end_calculated = self._range_ends(shard_number)
        assert range_ends == range_end_calculated, (range_ends, range_end_calculated, range_begins, shard_number)

        return shard_number


WebFreshTier = _PrefixLessJupiterTier(
    name='WebFreshTier',
    groups_count=10,
    shards_in_group=18,
    shard_size_gb=60,
)

CallistoSlotsTier = _PrefixLessJupiterTier(
    name='CallistoSlotsTier0',
    groups_count=11,
    shards_in_group=18,
    shard_size_gb=0,
)

PlatinumTier0 = _JupiterTier(
    name='PlatinumTier0',
    shards_in_group=180,
    shard_size_gb=5,
    prefix='primus',
)

WebTier0 = _JupiterTier(
    name='WebTier0',
    groups_count=1,
    shards_in_group=72,
    shard_size_gb=300,
    prefix='primus',
)

AttributeWebTier0 = _JupiterTier(
    name='AttributeWebTier0',
    groups_count=WebTier0.groups_count,
    shards_in_group=WebTier0.shards_in_group,
    shard_size_gb=300,
    prefix='primus',
)

MappingWebTier0 = _JupiterTier(
    name='MappingWebTier0',
    groups_count=WebTier0.groups_count,
    shards_in_group=WebTier0.shards_in_group,
    shard_size_gb=60,
    prefix='primus',
)

EmbeddingWebTier0 = _PrefixLessJupiterTier(
    name='EmbeddingWebTier0',
    groups_count=72,
    shards_in_group=1,
    shard_size_gb=140,
)

InvertedIndexWebTier0 = _PrefixLessJupiterTier(
    name='InvertedIndexWebTier0',
    groups_count=72,
    shards_in_group=54,
    shard_size_gb=90,
)

KeyInvWebTier0 = _JupiterTier(
    name='KeyInvWebTier0',
    groups_count=1,
    shards_in_group=72,
    shard_size_gb=75,
    prefix='primus',
)

WebTier0Mini = _JupiterTier(
    name='WebTier0Mini',
    groups_count=1,
    shards_in_group=1,
    shard_size_gb=30,
    prefix='primus',
)

AttributeWebTier0Mini = _JupiterTier(
    name='AttributeWebTier0Mini',
    groups_count=WebTier0Mini.groups_count,
    shards_in_group=WebTier0Mini.shards_in_group,
    shard_size_gb=30,
    prefix='primus',
)

EmbeddingWebTier0Mini = _PrefixLessJupiterTier(
    name='EmbeddingWebTier0Mini',
    groups_count=1,
    shards_in_group=1,
    shard_size_gb=14,
)

InvertedIndexWebTier0Mini = _PrefixLessJupiterTier(
    name='InvertedIndexWebTier0Mini',
    groups_count=1,
    shards_in_group=54,
    shard_size_gb=9,
)

KeyInvWebTier0Mini = _JupiterTier(
    name='KeyInvWebTier0Mini',
    groups_count=1,
    shards_in_group=1,
    shard_size_gb=7,
    prefix='primus',
)

WebTier1 = _JupiterTier(
    name='WebTier1',
    shards_in_group=1476,
    shard_size_gb=85,
    prefix='primus',
)


@memoize.get_cache(records_limit=1)
def get_erasure_test_tier():
    global TIERS
    if 'WebTier1' in TIERS:
        del TIERS['WebTier1']
    return _JupiterTier(
        name='WebTier1',
        shards_in_group=1,
        shard_size_gb=800,
        prefix='primus',
    )


MsUserData = _JupiterMsTier(
    name='MsuseardataJupiterTier0',
    shards_in_group=1,
    shard_size_gb=45,
    prefix='rearr',
)

JudTier = _JupiterTier(
    name='JudTier',
    shards_in_group=54,
    shard_size_gb=30,
    prefix='primus',
)


EmbeddingWebTier1 = _PrefixLessJupiterTier(
    name='EmbeddingWebTier1',
    shards_in_group=1,
    shard_size_gb=120,
    groups_count=82,
)

InvertedIndexWebTier1 = _PrefixLessJupiterTier(
    name='InvertedIndexWebTier1',
    shards_in_group=54,
    shard_size_gb=50,
    groups_count=82,
)


# Images

ImgTier0 = _ImgVideoTier(
    name='ImgTier0',
    shards_in_group=432,
    shard_size_gb=105,
    prefix='imgsidx',
)

ImgTier1 = _ImgVideoTier(
    name='ImgTier1',
    shards_in_group=432,
    shard_size_gb=165,
    prefix='imgsidxt1',
)

ImgInvIndexTier0 = _ImgVideoGroupedTier(
    name='InvertedIndexImgTier0',
    groups_count=24,
    shards_in_group=18,
    shard_size_gb=3.5,
    prefix='imgsinverted',
)

ImgEmbeddingTier0 = _ImgVideoTier(
    name='EmbeddingImgTier0',
    shards_in_group=24,
    shard_size_gb=13,
    prefix='imgsembedding',
)

ImgMmetaTier0 = _ImgVideoTier(
    name='ImgMmetaTier0',
    shards_in_group=1,
    shard_size_gb=12,
    prefix='imgmmeta',
)

ImgRIMTier = _ImgVideoTier(
    name='ImgsRim3k',
    shards_in_group=3000,
    shard_size_gb=7,
    prefix='imgsrim3k',
)

ImgCommercialTier0 = _ImgVideoTier(
    name='ImgCommercialTier0',
    shards_in_group=25,
    shard_size_gb=120,
    prefix='img_commercial',
)


class _ImgThumbTier(_ImgVideoTier):
    class _Shard(_ImgVideoTier._Shard):
        shard_template = '{prefix}-{shard_number:04}-{date}-{time}'
        number_template = '{shard_number:04}'


class _ImgThumbTier20k(_ImgVideoTier):
    class _Shard(_ImgVideoTier._Shard):
        shard_template = '{prefix}-{shard_number:05}-{date}-{time}'
        number_template = '{shard_number:05}'


ImtubWideTier0 = _ImgThumbTier(
    name='ImtubWideTier0',
    shards_in_group=2000,
    shard_size_gb=510,
    prefix='imgsth_wide',
)

ImtubLargeTier0 = _ImgThumbTier(
    name='ImtubLargeTier0',
    shards_in_group=2000,
    shard_size_gb=230,
    prefix='imglth',
)

ImtubWide20kTier = _ImgThumbTier20k(
    name='ImtubWide20kTier',
    shards_in_group=20000,
    shard_size_gb=50,
    prefix='imgsth_wide_20k'
)


# Video

VideoTier0 = _ImgVideoTier(
    name='VideoTier0',
    shards_in_group=252,
    shard_size_gb=85,
    prefix='vidsidx',
)

VidMmetaTier0 = _ImgVideoTier(
    name='VidMmetaTier0',
    shards_in_group=1,
    shard_size_gb=1,
    prefix='vidmmeta',
)

VideoPlatinum = _ImgVideoTier(
    name='VideoPlatinum',
    shards_in_group=216,
    shard_size_gb=8,
    prefix='vidsidxpt',
)

VideoEmbeddingPlatinum = _ImgVideoTier(
    name='VideoEmbeddingPlatinum',
    shards_in_group=12,
    shard_size_gb=2,
    prefix='VideoEmbeddingPlatinum',
)

VideoEmbeddingTier0 = _ImgVideoTier(
    name='VideoEmbeddingTier0',
    shards_in_group=14,
    shard_size_gb=27,
    prefix='VideoEmbeddingTier0',
)

VideoInvertedPlatinum = _ImgVideoGroupedTier(
    name='VideoInvertedPlatinum',
    shards_in_group=18,
    shard_size_gb=0.5,
    groups_count=12,
    prefix='VideoInvertedPlatinum',
)

VideoInvertedTier0 = _ImgVideoGroupedTier(
    name='VideoInvertedTier0',
    shards_in_group=18,
    shard_size_gb=1.5,
    groups_count=14,
    prefix='VideoInvertedTier0',
)

# Gemini

GeminiTier = _GeminiTier(
    name='GeminiTier',
    shards_in_group=2132,
    shard_size_gb=16,
)

CastorTier = _GeminiTier(
    name='CastorTier',
    shards_in_group=1,
    shard_size_gb=15,
)

# Quick
#
SaasFrozen3dayTier = _SaasFrozen3dayTier(
    name='SaasFrozen3dayTier0',
    shards_in_group=10,
    shard_size_gb=65,
)

# Image saas quick
ImgsSaasQuickTier = _SaasFrozen3dayTier(
    name='ImgsSaasQuickTier0',
    shards_in_group=18,
    shard_size_gb=5,
)

# Video saas quick
VideoSaasQuickTier = _SaasFrozen3dayTier(
    name='SaasFrozenVideoQuickTier0',
    shards_in_group=12,
    shard_size_gb=12,
)

NewsSaasQuickTier = _SaasFrozen3dayTier(
    name='SaasNewsTier0',
    shards_in_group=1,
    shard_size_gb=3,
)


def get_tier_by_name(name):
    return TIERS[name]
