import infra.callisto.protos.deploy.tables_pb2 as tables_pb2
import search.plutonium.deploy.proto.rescan_pb2 as rescan_pb2


class _AbstractTarget(object):
    namespace = None

    def __init__(self, stream, snapshot, resource_spec):
        self._stream = stream
        self._snapshot = snapshot
        self._resource_spec = resource_spec

    def get_localpath(self):
        return '{}/{}'.format(
            self._stream,
            self._snapshot,
        )

    def get_labels(self):
        return rescan_pb2.TResourceLabels(Label={
            'Stream': self._stream,
            'SnapshotId': self._snapshot,
        })

    @classmethod
    def get_snapshot(cls, target):
        labels = target.ResourceLabels.Label
        if set(['Stream', 'SnapshotId']).issubset(labels.keys()):
            return (
                labels['Stream'],
                labels['SnapshotId']
            )
        # FALLBACK
        path = target.LocalPath.split('/')
        assert len(path) > 1
        return (path[0], path[1])

    def get_targets(self, pod_ids):
        assert self.namespace

        localpath = self.get_localpath()
        labels = self.get_labels()

        return [
            tables_pb2.TPodTarget(
                PodId=pod,
                Namespace=self.namespace,
                LocalPath=localpath,
                ResourceSpec=self._resource_spec,
                ResourceLabels=labels
            )
            for pod in pod_ids
        ]


class MappingsTarget(_AbstractTarget):
    namespace = 'mappings'


class TopologyTarget(_AbstractTarget):
    namespace = 'topology'

    def __init__(self, stream, snapshot, resource_spec, revision, state_ts):
        super(TopologyTarget, self).__init__(stream, snapshot, resource_spec)
        self._revision = revision
        self._state_ts = state_ts

    def get_localpath(self):
        return '{}/{}/{}.{}.conf'.format(
            self._stream,
            self._snapshot,
            self.namespace,
            self._revision
        )

    def get_labels(self):
        resource_labels = super(TopologyTarget, self).get_labels()
        resource_labels.Label['Revision'] = str(self._revision)
        resource_labels.Label['Timestamp'] = str(self._state_ts)

        return resource_labels


class ItemTypeTarget(_AbstractTarget):
    namespace = 'metachunks'

    def __init__(self, stream, snapshot, resource_spec, item_type):
        super(ItemTypeTarget, self).__init__(stream, snapshot, resource_spec)
        self._item_type = item_type

    def get_localpath(self):
        return '{}/{}/{}'.format(
            self._stream,
            self._snapshot,
            self._item_type
        )

    def get_labels(self):
        resource_labels = super(ItemTypeTarget, self).get_labels()
        resource_labels.Label['ItemType'] = str(self._item_type)

        return resource_labels
