import json

from infra.yasm.unistat import global_unistat, AggregationType


class HoleAggregationType:
    def __init__(self, aggregation_type):
        self.__aggregation_type = aggregation_type
        self.__unistat_aggregation_type = aggregation_type
        if aggregation_type in (AggregationType.Min, AggregationType.Max, AggregationType.Average):
            self.__unistat_aggregation_type = AggregationType.LastValue

    def get_type(self):
        return self.__aggregation_type

    def get_unistat_type(self):
        return self.__unistat_aggregation_type


class FloatHole:
    def __init__(self, signal_name, suffix, aggregation_type, tags=None):
        self.__hole_aggregation_type = HoleAggregationType(aggregation_type)
        self._hole = global_unistat.create_float(
            name=signal_name,
            tags=tags if tags else [],
            suffix=suffix,
            aggregation_type=self.__hole_aggregation_type.get_unistat_type(),
        )
        self._reset_aggregate_values()

    def _reset_aggregate_values(self):
        self._values_min = 2 ** 32
        self._values_max = 0
        self._values_sum = 0
        self._values_count = 0

    def push(self, value):
        if self.__hole_aggregation_type.get_type() == AggregationType.Min:
            self._values_min = min(self._values_min, value)
            self._values_count += 1
        elif self.__hole_aggregation_type.get_type() == AggregationType.Max:
            self._values_max = max(self._values_max, value)
            self._values_count += 1
        elif self.__hole_aggregation_type.get_type() == AggregationType.Average:
            self._values_sum += value
            self._values_count += 1
        else:
            self._hole.push(value)

    def flush_aggregate(self):
        if not self._values_count:
            return

        if self.__hole_aggregation_type.get_type() == AggregationType.Min:
            self._hole.push(self._values_min)
        elif self.__hole_aggregation_type.get_type() == AggregationType.Max:
            self._hole.push(self._values_max)
        elif self.__hole_aggregation_type.get_type() == AggregationType.Average:
            self._hole.push(self._values_sum / self._values_count)

        self._reset_aggregate_values()


class Unistat:
    def __init__(self):
        self._holes = {}
        self._signal_prefix_to_suffixes = {}

    def drill_float(self, signal_prefix, suffix, aggregation_type=AggregationType.LastValue, tags=None):
        self._holes['{}_{}'.format(signal_prefix, suffix)] = FloatHole(
            signal_prefix,
            suffix,
            aggregation_type,
            tags
        )

    def drill_float_aggregation(self, signal_prefix, tags=None, delta=False):
        names_suffixes = ['min', 'max', 'aver']
        suffixes = ['dnnn', 'dxxx', 'dvvv'] if delta else ['annn', 'axxx', 'avvv']
        aggregation_types = [AggregationType.Min, AggregationType.Max, AggregationType.Average]
        for name_suffix, suffix, aggregation_type in zip(names_suffixes, suffixes, aggregation_types):
            self.drill_float('{}_{}'.format(signal_prefix, name_suffix), suffix, aggregation_type, tags)
            self._signal_prefix_to_suffixes[signal_prefix] = zip(names_suffixes, suffixes)

    def push(self, signal_name, value):
        self._holes[signal_name].push(value)

    def push_aggregation(self, signal_prefix, value):
        for name_suffix, suffix in self._signal_prefix_to_suffixes[signal_prefix]:
            self.push('{}_{}_{}'.format(signal_prefix, name_suffix, suffix), value)

    def flush_aggregate(self, signal_name):
        self._holes[signal_name].flush_aggregate()

    def flush_aggregation(self, signal_prefix):
        for name_suffix, suffix in self._signal_prefix_to_suffixes[signal_prefix]:
            self.flush_aggregate('{}_{}_{}'.format(signal_prefix, name_suffix, suffix))

    def to_list(self):
        return json.loads(global_unistat.to_json())
